<?php

namespace SolveX\Support;

class Dotenv
{
    protected $filePath;

    public function __construct($path, $file = '.env')
    {
        $this->filePath = rtrim($path, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $file;
    }

    /**
     * Parses .env file into an array of key => value pairs (returned value).
     * Handles 'false', 'true' values.
     * Does *NOT* affect the environment ($_ENV/$_SERVER/getenv).
     */
    public function load()
    {
        $lines = $this->readLinesFromFile($this->filePath);

        $env = [];

        foreach ($lines as $line) {
            if ($this->isComment($line))
                continue;

            list($key, $value) = $this->splitLine($line);

            $value = $this->removeQuotes($value);

            if ($value === 'false') $value = false;
            if ($value === 'true')  $value = true;

            $env[$key] = $value;
        }

        return $env;
    }

    protected function readLinesFromFile($filePath) {
        // Read file into an array of lines with auto-detected line endings.
        $autodetect = ini_get('auto_detect_line_endings');
        ini_set('auto_detect_line_endings', '1');
        $lines = file($filePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        ini_set('auto_detect_line_endings', $autodetect);
        return $lines;
    }

    protected function isComment($line)
    {
        return strpos(trim($line), '#') === 0;
    }

    protected function splitLine($line)
    {
        list($key, $value) = explode('=', $line, 2);
        $key = trim($key);
        $value = trim($value);
        return [$key, $value];
    }

    protected function removeQuotes($value)
    {
        if (strlen($value) === 0)
            return $value;

        $firstChar = $value[0];
        $lastChar = substr($value, -1);
        $singleQuoted = ($firstChar === "'" && $lastChar === "'");
        $doubleQuoted = ($firstChar === '"' && $lastChar === '"');
        if ($singleQuoted || $doubleQuoted) {
            $value = substr($value, 1, strlen($value)-2);
        }

        return $value;
    }
}
