<?php

namespace SolveX\Exceptions;

use SolveX\Interfaces\ExceptionHandlerInterface;
use SolveX\Exceptions\HttpException;
use SolveX\Exceptions\NotFoundHttpException;
use Exception;
use Log;

class Handler implements ExceptionHandlerInterface
{
    /**
     * Report or log an exception (or redirect to homepage or something else).
     *
     * @param  \Exception  $e
     * @return void
     */
    public function handle(Exception $exception)
    {
        if ($this->isCli()) {
            ini_set('html_errors', false);
            var_dump($exception);
            return;
        }

        if ($this->isAjax())
            $this->renderAjaxError($exception);
        else
            $this->renderHtmlError($exception);
    }


    /**
     * Is the application running in command-line?
     */
    protected function isCli()
    {
        return php_sapi_name() === 'cli';
    }


    /**
     * Are we handling an AJAX request?
     */
    protected function isAjax()
    {
        if (empty($_SERVER['HTTP_X_REQUESTED_WITH']))
            return false;

        return (strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest');
    }


    /**
     * Casts an object into an array, including private and protected properties.
     */
    protected function objectToArray($object)
    {
        $arr = (array) $object;
        $filtered = [];

        // Non-public properties have \0 and * characters after (array) cast. Let's remove them.
        foreach ($arr as $key => $value)
            $filtered[str_replace(["\0", '*'], '', $key)] = $value;

        return $filtered;
    }


    /**
     * In case of an AJAX request, respond with JSON.
     *
     * @todo standardize JSON response!
     */
    protected function renderAjaxError($exception)
    {
        if ($exception instanceof HttpException) {
            $status = $exception->getStatusCode();
            $error = $exception->getMessage();
        }
        else {
            $status = 500;
            $error = 'There was an error. Please notify site administrators.';
            if (env('DEBUG', false))
                $error = $this->objectToArray($exception);
        }

        header('Content-Type: application/json');
        http_response_code($status);
        echo(json_encode([
            'errors' => [$error],
        ]));
    }


    /**
     * Use tracy to output nicely formatted error text (in debug mode).
     *
     * @link https://github.com/nette/tracy
     */
    protected function renderHtmlError($exception)
    {
        ob_clean();

        if (! env('DEBUG', false)) {
            echo 'There was an error. Please notify site administrators.';
            exit();
        }

        \Tracy\Debugger::enable(\Tracy\Debugger::DEVELOPMENT);
        \Tracy\Debugger::exceptionHandler($exception);
    }
}
