<?php

namespace SolveX\Database;

use SolveX\Interfaces\DatabaseInterface;

class MySqli implements DatabaseInterface
{
    protected $mysqli = null;

    public function __construct($config)
    {
        $host     = $config['host'];
        $user     = $config['user'];
        $password = $config['pass'];
        $database = $config['database'];

        $this->mysqli = mysqli_connect($host, $user, $password, $database);

        if (mysqli_connect_errno())
            throw new RuntimeException('mysqli connection failed: ' . mysqli_connect_error());

        // Throw exceptions,
        // http://stackoverflow.com/questions/14578243/turning-query-errors-to-exceptions-in-mysqli
        // MYSQLI_REPORT_INDEX warns when no index is used!
        mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

        // Use utf8mb4 (proper up-to-4-bytes multibyte utf8) ?
        // https://mathiasbynens.be/notes/mysql-utf8mb4
        // It is important we call mysqli_set_charset! See http://stackoverflow.com/a/12202218/238790
        mysqli_set_charset($this->mysqli, 'utf8');
    }


    /**
     * Run a select statement and return a single result.
     *
     * Example:
     * <code>
     * $id = $_POST['id']; // No need for escaping, it's done by DB.
     * $user = DB::selectOne('SELECT age, name FROM user WHERE id = ?', [$id]);
     * // Instead of question mark placeholders you can also use named placeholders:
     * $user = DB::selectOne('SELECT age, name FROM user WHERE id = :id', ['id' => $id]);
     * $newAge = $user->age + 1;
     * </code>
     *
     * If retrieving one field only, selectOne() returns just that field value, not a row object:
     * <code>
     * $age = DB::selectOne('SELECT age FROM user WHERE id = ?', [$id]);
     * </code>
     *
     * @param string $sql Query to run.
     * @param array $bindings Bindings.
     *
     * @return mixed An instance of stdClass class, string, or null if no result.
     *
     * @link http://jpauli.github.io/2014/07/21/php-and-mysql-communication-mysqlnd.html
     * @link http://blog.ulf-wendel.de/2014/php-5-7-mysqlnd-memory-optimizations/
     *
     */
    public function selectOne($sql, $bindings = [])
    {
        $result = mysqli_query($this->mysqli, $this->insertBindings($sql, $bindings));

        $numFields = mysqli_num_fields($result);
        if ($numFields === 1)
            $obj = mysqli_fetch_row($result)[0];
        else
            $obj = mysqli_fetch_object($result);

        mysqli_free_result($result);
        return $obj;
    }


    /**
     * Run a select statement against the database. Returns an array of objects.
     *
     * Example:
     * <code>
     * $age = $_GET['age']; // No need for escaping, it's done by DB.
     * $users = DB::select('SELECT id, name FROM user WHERE age < :age', ['age' => $age]);
     * foreach ($users as $user) {
     *     echo $user->id;
     * }
     * </code>
     *
     * If retrieving one field only (column), select() returns just that field values, not row objects:
     * <code>
     * $ages = DB::select('SELECT age FROM user WHERE parent = ?', [$parentId]);
     * // e.g. $ages = ['12', '42']
     * </code>
     *
     * @param  string  $sql       Query to run.
     * @param  array   $bindings  Bindings.
     * @return array              Array of stdClass instances / strings.
     *
     * @link http://jpauli.github.io/2014/07/21/php-and-mysql-communication-mysqlnd.html
     * @link http://blog.ulf-wendel.de/2014/php-5-7-mysqlnd-memory-optimizations/
     */
    public function select($sql, $bindings = [])
    {
        $result = mysqli_query($this->mysqli, $this->insertBindings($sql, $bindings));
        $numFields = mysqli_num_fields($result);
        $resultset = [];

        if ($numFields === 1) {
            while ($row = mysqli_fetch_row($result))
                $resultset[] = $row[0];
        }
        else {
            while ($obj = mysqli_fetch_object($result))
                $resultset[] = $obj;
        }

        mysqli_free_result($result);
        return $resultset;
    }


    /**
     * An insert statement.
     *
     * Example of an INSERT query. Notice NOW() is in the query itself,
     * not the bindings.
     * <code>
     * DB::insert('INSERT INTO db2.log(datetime, type, severity, message)
     *             VALUES (NOW(), :type, :severity, :message)', [
     *                 'type' => 2,
     *                 'severity' => 3,
     *                 'message' => 'User age updated'
     *             ]);
     * </code>
     *
     * @param  string  $sql       Query to run.
     * @param  array   $bindings  Bindings.
     * @return bool               Success/failure.
     */
    public function insert($sql, $bindings = [])
    {
        return mysqli_query($this->mysqli, $this->insertBindings($sql, $bindings));
    }


    /**
     * An update statement.
     *
     * @param  string  $sql       Query to run.
     * @param  array   $bindings  Bindings.
     * @return int                Num of affected rows.
     */
    public function update($sql, $bindings = [])
    {
        return $this->affectingStatement($sql, $bindings);
    }


    /**
     * A delete statement.
     *
     * @param  string  $sql       Query to run.
     * @param  array   $bindings  Bindings.
     * @return int                Num of affected rows.
     */
    public function delete($sql, $bindings = [])
    {
        return $this->affectingStatement($sql, $bindings);
    }


    /**
     * For update or delete statements, we want to get the number of rows affected
     * by the statement and return that back.
     */
    protected function affectingStatement($sql, $bindings)
    {
        mysqli_query($this->mysqli, $this->insertBindings($sql, $bindings));

        return mysqli_affected_rows($this->mysqli);
    }


    /**
     * Plain query.
     *
     * @param string $sql
     * @param array $bindings
     *
     * @return mixed mysqli_query return value (depends on the query itself).
     */
    public function query($sql, $bindings = [])
    {
        return mysqli_query($this->mysqli, $this->insertBindings($sql, $bindings));
    }


    /**
     * Runs database queries in a transaction.
     *
     * Any exception in the closure rollbacks the transaction. Otherwise the transaction
     * is automatically committed.
     * Exceptions thrown are re-thrown after the rollback so that they can be handled
     * how the developer sees fit.
     *
     * Example:
     * <code>
     * DB::transaction(function() {
     *     DB::update('UPDATE account SET balance = balance - ? WHERE id = ?', [500, 12]);
     *     DB::update('UPDATE account SET balance = balance + ? WHERE id = ?', [500, 42]);
     * });
     * </code>
     *
     * Another example with locking:
     * <code>
     * DB::transaction(function() {
     *     $room = DB::selectOne('SELECT id, status FROM room WHERE empty = 1 FOR UPDATE');
     *     // SELECT FOR UPDATE reads the latest available data, setting exclusive locks
     *     // on each row it reads. Only applies in a transaction.
     *     DB::update('UPDATE room SET status = 2 WHERE id = ?', [$room->id]);
     * });
     * </code>
     *
     * Transactions are not available on MyISAM tables (use InnoDB)!
     *
     * Some statements cannot be rolled back (they implicitly commit). In general, these include data definition
     * language (DDL) statements, such as those that create or drop databases,
     * those that create, drop, or alter tables or stored routines.
     *
     * When the script ends or when a connection is about to be closed,
     * if you have an outstanding transaction, it will automatically rolled back.
     * This is a safety measure to help avoid inconsistency in the cases
     * where the script terminates unexpectedly--if you didn't explicitly
     * commit the transaction, then it is assumed that something went awry,
     * so the rollback is performed for the safety of your data.
     *
     * Nested transactions are not supported.
     *
     * MySQL's default transaction isolation level is REPEATABLE READ.
     * This means updates/inserts/deletes (even committed) from concurrent transactions
     * are not visible to this transaction.
     * For protection from concurrent transactions,
     * consider Locking reads (e.g. SELECT FOR UPDATE).
     * http://dev.mysql.com/doc/refman/5.7/en/innodb-locking-reads.html
     * http://dba.stackexchange.com/a/15864
     * http://stackoverflow.com/questions/10935850/when-to-use-select-for-update
     *
     * @param  Closure  $closure  Closure to run in a transaction.
     *
     * @return mixed Return value of your $closure.
     */
    public function transaction(\Closure $closure)
    {
        mysqli_autocommit($this->mysqli, false);

        $exceptionOrNull = null;

        try {
            $result = $closure();

            mysqli_commit($this->mysqli);
        }
        catch (\Exception $exception) {
            // Rollback and re-throw.
            //
            mysqli_rollback($this->mysqli);

            $exceptionOrNull = $exception;
        }

        mysqli_autocommit($this->mysqli, true); // Turn autocommit back on!

        // finally is PHP 5.5+
        if ($exceptionOrNull)
            throw $exceptionOrNull;

        return $result;
    }


    /**
     * Escapes *and quotes* a string.
     * If $what is null, returns string 'null'.
     * If $what is false, returns string '0'.
     *
     * @param mixed $what
     *
     * @return string
     */
    protected function quote($what)
    {
        if ($what === null)
            return 'null';

        if ($what === false)
            return '0';

        return '"' . mysqli_real_escape_string($this->mysqli, $what) . '"';
    }


    protected function insertBindings($sql, $bindings)
    {
        if (empty($bindings))
            return $sql;

        // We support both question mark ('?') placeholders and
        // named placeholders (':id'), but not in the same query.

        if (is_int(key($bindings)))
            return $this->insertQuestionPlaceholderBindings($sql, $bindings);
        else
            return $this->insertNamedPlaceholderBindings($sql, $bindings);
    }


    protected function insertQuestionPlaceholderBindings($sql, $bindings)
    {
        $finalSql = ''; // Stitched together. Question mark replaced with quoted values from $bindings.
        $offset = 0;

        foreach ($bindings as $value) {
            $newOffset = strpos($sql, '?', $offset); // Next '?'

            if ($newOffset === false)
                throw new \RuntimeException('Not enough placeholders!');

            $finalSql .= substr($sql, $offset, $newOffset-$offset);

            $finalSql .= $this->quoteArray($value);

            $offset = $newOffset+1; // Skip question mark placeholder.
        }

        // Last part (after the last placeholder).
        $finalSql .= substr($sql, $offset);

        return $finalSql;
    }


    protected function quoteArray($arr)
    {
        // If not an array, just quote it.
        if (! is_array($arr))
            return $this->quote($arr);

        // Else, quote elements inside the array (first-level only),
        // and wrap quoted elements with parenthesis.

        $numElems = count($arr);
        if ($numElems === 0)
            return '()';

        $quoted = '(';
        for ($i = 0; $i < $numElems-1; $i++)
            $quoted .= $this->quote($arr[$i]) . ',';

        $quoted .= $this->quote($arr[$i]) . ')';
        return $quoted;
    }


    protected function insertNamedPlaceholderBindings($sql, $bindings)
    {
        return preg_replace_callback(
            '/:\w+/',
            function($matches) use (&$bindings) {
                $bind = $matches[0];
                if (! array_key_exists($bind, $bindings)) // Support both [':id' => 42]  and  ['id' => 42]
                    $bind = substr($bind, 1);

                if (! array_key_exists($bind, $bindings))
                    throw new \RuntimeException('Placeholder ' . $bind . ' not found!');

                $value = $bindings[$bind];
                $replacement = $this->quoteArray($value);

                unset($bindings[$bind]); // Imitate PDO behaviour - named placeholder cannot be repeated.
                return $replacement;
            },
            $sql
        );
    }
}
