<?php

/**
 * Trigger all errors, log them, but don't just directly output them.
 */
ini_set('display_errors', 'off');
ini_set('display_startup_errors', false);
ini_set('log_errors', true);
ini_set('html_errors', true);
error_reporting(E_ALL);

/**
 * Enable assert.
 *
 * @todo http://php.net/manual/en/function.assert-options.php
 */
assert_options(ASSERT_ACTIVE, true);
assert_options(ASSERT_WARNING, true);

/**
 * Translate PHP error into an ErrorException.
 *
 */
set_error_handler(function ($level, $message, $file = '', $line = 0) {
    if (error_reporting() & $level)
        throw new ErrorException($message, 0, $level, $file, $line);
});

/**
 * Handle an uncaught exception from the application.
 *
 * @link http://php.net/manual/en/spl.exceptions.php
 */

set_exception_handler('handleException');

/**
 * Catch fatal errors.
 *
 * @link http://stackoverflow.com/questions/15598751/how-to-catch-require-once-include-once-exception
 */
register_shutdown_function(function () {
    $error = error_get_last();

    if (is_null($error))
        return;

    if (! in_array($error['type'], [E_ERROR, E_CORE_ERROR, E_COMPILE_ERROR, E_PARSE])) // If not a fatal error
        return;

    $exception = new Symfony\Component\Debug\Exception\FatalErrorException(
        $error['message'], $error['type'], 0, $error['file'], $error['line'], 0
    );

    handleException($exception);
});

/**
 * Delegates exception/wrapped error to a registered exception handler (customizable).
 */
function handleException($exception)
{
    if (! $exception instanceof Exception)
        $exception = new Symfony\Component\Debug\Exception\FatalThrowableError($exception);

    $container = app();
    $exceptionHandler = $container->make(SolveX\Interfaces\ExceptionHandlerInterface::class);
    $exceptionHandler->handle($exception);
}

/**
 * Get value from the environment. Handles 'true' and 'false' values correctly (string 'false' is truthy in PHP!)
 *
 * <code>
 * env('DB_HOST') // whatever is under DB_HOST in .env file
 * </code>
 */
function env($key, $default = null)
{
    static $env = null;
    if ($env === null) {
        $env = (new SolveX\Support\WarpitConfig())->load();
    }
    if (isset($env[$key]))
        return $env[$key];

    if ($default !== null)
        return $default;

    throw new RuntimeException('Environment var ' . $key . ' not found (.env file)!');
}

/**
 * Setup composer autoload here.
 *
 * @link https://getcomposer.org/
 */
if (! is_readable(__DIR__ . '/../vendor/autoload.php')) {
    echo 'vendor/autoload.php does not exist! Did you run composer install?';
    exit;
}

require __DIR__ . '/../vendor/autoload.php';

/**
 * Facade autoloader.
 *
 * This autoloader resolves facades, e.g. DB, Session, etc.
 *
 * Tweet-sized explanation:
 *
 * DB class is aliased to SolveX\Facades\DB
 * => SolveX\Facades\DB extends SolveX\Facades\Facade
 * => SolveX\Facades\Facade::__callStatic delegates static function calls
 *    to an instance of a class created with $container->make('DB')
 */
spl_autoload_register(function ($className) {
    $container = app();
    $facades = $container['facades'];

    if (isset($facades[$className])) {
        $instance = $container->make($className);
        SolveX\Facades\Facade::setInstance($facades[$className], $instance);
        class_alias($facades[$className], $className);
    }
});

/**
 * Builds (once) and returns a dependency injection container.
 *
 * @link https://github.com/illuminate/container
 */
function app()
{
    static $container = null;

    if ($container === null)
        require __DIR__ . '/config.php';

    return $container;
}

/**
 * Register facades & interfaces (into a "container").
 */
app();
