<?php

namespace App\Services;

use App\Repositories\QuestionRepository;
use App\Repositories\ProjectRepository;
use App\Repositories\StructureTableRepository;

use App\Entities\Answer;
use App\Entities\Question;
use App\Entities\QuestionType;
use App\Entities\LanguageTextType;
use App\Services\StructureTableService;
use App\Services\AnswersConvertService;
use App\Services\QuestionService;

use App\Exceptions\RuntimeException;

class AnswerService
{
    public function __construct(
      QuestionRepository    $questionRepository,
      ProjectRepository     $projectRepository,
      WarpitConfigService   $warpitConfigService,
      StructureTableService $structureTableService,
      AnswersConvertService $answersConvertService,
      QuestionService       $questionService
      ) {
        $this->questionRepository    = $questionRepository;
        $this->projectRepository     = $projectRepository;
        $this->warpitConfigService   = $warpitConfigService;
        $this->structureTableService = $structureTableService;
        $this->answersConvertService = $answersConvertService;
        $this->questionService       = $questionService;
    }

    public function addAnswer($questionName, $projectName, $value, $label, $languageId)
    {
        $question   = $this->questionService->getIfExists($questionName, $projectName, $languageId);

        $question->answersArray = $this->answersConvertService->toArray($question->answersString);

        $answer = new Answer;
        $answer->id    = $value;
        $answer->label = $label;

        $newColumns = $question->addAnswerAndGetNewColumns($answer);

        if($newColumns === false){
            throw new RuntimeException("Can't add answer!", 400);
        }

        $question->answersString = $this->answersConvertService->toString($question->answersArray);

        $this->addAnswersString($question->id, $projectName, $question->answersString, $languageId);

        if(count($newColumns) > 0){
            $this->structureTableService->addColumns($projectName, $newColumns);
        }

        return $question->answersArray;
    }

    public function deleteAnswer($questionName, $projectName, $value, $languageId)
    {
        $question = $this->questionService->getIfExists($questionName, $projectName, $languageId);

        $question->answersArray = $this->answersConvertService->toArray($question->answersString);
        $columnsToDrop = $question->removeAnswerAndGetColumnsForDrop($value);

        if($columnsToDrop === false){
          throw new RuntimeException("Can't delete answer", 400);
        }

        if(count($columnsToDrop) > 0) {
            $this->structureTableService->dropColumns($projectName, $columnsToDrop);
        }

        $answersString = $this->answersConvertService->toString($question->answersArray);
        $this->addAnswersString($question->id, $projectName, $answersString, $languageId);

        return $question->answersArray;
    }


    public function addAnswersString($questionId, $projectName, $answersString, $languageId)
    {
        if ($answersString === null) {
            return $this->questionRepository->deleteFromLanguageTable(
              $questionId,
              $projectName,
              LanguageTextType::ANSWER_STRING,
              $languageId
            );
        }

        $oldRecord = $this->questionRepository->getFromLanguageTable(
          $questionId,
          $projectName,
          LanguageTextType::ANSWER_STRING,
          $languageId
        );

        if ($oldRecord === null) {
            return $this->questionRepository->insertIntoLanguageTable(
              $questionId,
              $projectName,
              LanguageTextType::ANSWER_STRING,
              $answersString,
              $languageId
            );
        }

        return $this->questionRepository->updateLanguageTable(
          $questionId,
          $projectName,
          LanguageTextType::ANSWER_STRING,
          $answersString,
          $languageId
        );
    }

    public function getAnswersString($questionName, $projectName, $languageId)
    {
        $questionId = $this->questionService->getIfExists($questionName, $projectName, $languageId)->id;

        return $this
                ->questionRepository
                ->getFromLanguageTable($questionId, $projectName, LanguageTextType::ANSWER_STRING, $languageId);
    }

    public function getAnswers($questionName, $projectName, $languageId)
    {
        $answersString = $this->getAnswersString($questionName, $projectName, $languageId);

        return $this->answersConvertService->toArray($answersString);
    }
}
