<?php

namespace App\Repositories;

class ProjectRepository extends BaseRepository
{

    public function get($projectId)
    {
        return $this
            ->webcatiBaseConnection
            ->table('_Projects')
            ->select(['id','proj_name','naziv_proj'])
            ->where('id', $projectId)
            ->first();
    }

    public function createProjectTable($name)
    {
        $builder = $this->webcatiConnection->getSchemaBuilder();

        $builder->create($name, function($table)
        {
            $table->increments('id');
            $table->integer('q_pos');
            $table->char('q_name', 20);
            $table->integer('tip_sql')->nullable();
            $table->char('drg_sql', 1)->nullable();;
            $table->integer('les_sql')->nullable();
            $table->text('goto_sql')->default('');
            $table->text('fil_sql')->default('');
            $table->text('con_sql')->default('');
            $table->integer('open_sql')->nullable();
            $table->integer('anno_sql')->nullable();
            $table->integer('attachAnswers')->nullable();
            $table->boolean('attachFalse')->nullable();
            $table->text('rot_sql')->nullable();
            $table->tinyInteger('srot_sql')->nullable();
            $table->tinyInteger('erot_sql')->nullable();
            $table->tinyInteger('srotLegend')->nullable();
            $table->tinyInteger('erotLegend')->nullable();
            $table->tinyInteger('srotAttributes')->nullable();
            $table->tinyInteger('erotAttributes')->nullable();
            $table->char('qrot_sql', 9)->nullable();
            $table->bigInteger('id_style')->nullable();
            $table->bigInteger('id_mask')->nullable();
            $table->tinyInteger('must_answer')->default(0);
            $table->tinyInteger('disable_question')->default(0);
            $table->tinyInteger('answers_filter')->default(0);
            $table->tinyInteger('variableTypeBase')->default(4);
            $table->tinyInteger('variableTypeOpen')->default(0);
            $table->char('open_answers', 250)->nullable();
            $table->char('open_legend', 250)->nullable();
            $table->char('open_attributes', 250)->nullable();
            $table->char('multi_nogo_answers', 250)->nullable();
            $table->char('multi_nogo_legends', 100)->nullable();
            $table->text('multilanguage_all')->nullable();
            $table->tinyInteger('qleave_js_alert')->default(0);
            $table->text('filter_answer')->nullable();
            $table->text('filter_legend')->nullable();
            $table->text('filter_attributes')->nullable();
            $table->tinyInteger('hasStructure')->default(0)->nullable();
            $table->tinyInteger('use_for_quotas')->default(0)->nullable();
            $table->tinyInteger('isFinal')->default(0)->nullable();
            $table->integer('id_showTogether')->default(0);
            $table->text('answerGroups')->nullable();
            $table->text('legendGroups')->nullable();
            $table->integer('id_plugin')->nullable();
            $table->integer('id_incentive')->nullable();
            $table->tinyInteger('incentive_possibilites')->default(0);
            $table->tinyInteger('allow_listening')->default(0);
            $table->integer('old_id');
            $table->integer('up_down_rotchk');
            $table->integer('up_down_rotvar');
            $table->integer('repeat_legend');
            $table->tinyInteger('autoSubmit')->default(0);
            $table->char('missing_values', 100)->nullable();
            $table->tinyInteger('open_end_type')->default(1)->comment('1 text box 2 text area');
            $table->tinyInteger('enter_submit')->default(1);
            $table->string('slider_color', 10)->nullable();
            $table->string('trace_color', 10)->nullable();
            $table->string('slider_shape', 50)->nullable();
            $table->string('slider_handle_image', 255)->nullable();
            $table->integer('email_app_group')->nullable();
            $table->integer('email_app_template')->nullable();
            $table->tinyInteger('encrypt')->default(1)->nullable();
            $table->tinyInteger('multi_type')
                  ->nullable()
                  ->default(1)
                  ->comment('1 column multi response 2 line multi response 3 line and column');
            $table->index('q_pos');
            $table->index('tip_sql');
            $table->unique('q_name');
            $table->engine = 'InnoDB';
        });
    }

    public function createTreeTable($name)
    {
      $treeTable = $name . '_tree';
      $this->webcatiConnection->getSchemaBuilder()->create($treeTable, function($table)
      {
        $table->increments('id');
        $table->integer('id_sup');
        $table->integer('id_question');
        $table->smallInteger('orderBy');
        $table->string('idLab', 100);
        $table->string('node_img', 100);
        $table->tinyInteger('is_leaf');
        $table->boolean('showTogether');
        $table->index('id_sup');
        $table->index('id_question');
        $table->index('orderBy');
        $table->engine = 'InnoDB';
      });
    }

    public function createScriptTable($name)
    {
        $scriptTable = $name . '_script';
        $this->webcatiConnection->getSchemaBuilder()->create($scriptTable, function($table)
        {
          $table->integer('id_question');
          $table->integer('id_event');
          $table->text('script');
          $table->primary(['id_question', 'id_event']);
          $table->engine = 'InnoDB';
        });
    }

    public function createLanguageTable($name)
    {
        $languageTable = $name . '_language';

        $this->webcatiConnection->getSchemaBuilder()->create($languageTable, function($table)
        {
          $table->increments('id');
          $table->integer('id_language');
          $table->integer('id_question');
          $table->integer('id_text_type');
          $table->text('lngText');
          $table->index('id_question');
          $table->unique(['id_language', 'id_question', 'id_text_type'], 'qlt_index');
          $table->engine = 'InnoDB';
        });
    }

    public function insertIntoProjectTable($project)
    {
          return $this
                  ->webcatiBaseConnection
                  ->table('_Projects')
                  ->insert($project);
    }

    public function insertIntoProjectTreeTable($name, $orderBy, $subFolder = 0)
    {
        $projectId = $this
                ->webcatiBaseConnection
                ->table('_ProjectsTree')
                ->insertGetId([
                    'id_sup'  => $subFolder,
                    'orderBy' => $orderBy,
                    'idLab'   => $name,
                    'is_leaf' => true,
                 ]);
        $this
          ->webcatiBaseConnection
          ->table('_ProjectsTree')
          ->where('id', $projectId)
          ->update(['id_project' => $projectId]);

        return $projectId;
    }

    public function insertLanguage($languageId, $projectId)
    {
      return $this->webcatiBaseConnection
                  ->table('_LanguageProjects')
                  ->insert([
                    'id_language' => $languageId,
                    'id_project'  => $projectId,
                  ]);
    }

    public function getMaxOrder()
    {
        return $this
                ->webcatiBaseConnection
                ->table('_ProjectsTree')
                ->select('orderBy')
                ->orderByDesc('orderBy')
                ->first()
                ->orderBy;
    }

    public function getProject($name)
    {
        $project = $this
                    ->webcatiBaseConnection
                    ->table('_Projects')
                    ->select('*')
                    ->where('proj_name', $name)
                    ->first();

        return $project;
    }

    public function getProjectId($name)
    {
        $project = $this
                    ->webcatiBaseConnection
                    ->table('_Projects')
                    ->select('id')
                    ->where('proj_name', $name)
                    ->first();
        return $project !== NULL?$project->id:$project;
    }

    public function getProjectName($id)
    {
        $project = $this
                  ->webcatiBaseConnection
                  ->table('_Projects')
                  ->select(['id','proj_name','naziv_proj'])
                  ->where('id', $id)
                  ->first();
        return $project !== NULL?$project->proj_name:$project;
    }





    public function dropProjectTable($name)
    {
        $builder = $this->webcatiConnection->getSchemaBuilder();
        $builder->dropIfExists($name);
    }

    public function dropProjectTreeTable($name)
    {
        $treeTable = $name . '_tree';
        $builder = $this->webcatiConnection->getSchemaBuilder();
        $builder->dropIfExists($treeTable);
    }

    public function dropProjectScriptTable($name)
    {
        $scriptTable = $name . '_script';
        $builder = $this->webcatiConnection->getSchemaBuilder();
        $builder->dropIfExists($scriptTable);
    }

    public function dropProjectLanguageTable($name)
    {
        $languageTable = $name . '_language';
        $builder = $this->webcatiConnection->getSchemaBuilder();
        $builder->dropIfExists($languageTable);
    }

    public function deleteFromProjectTable($id)
    {
        $this
          ->webcatiBaseConnection
          ->table('_Projects')
          ->where('id', $id)
          ->delete();
    }

    public function deleteFromProjectsTreeTable($id)
    {
        $this
          ->webcatiBaseConnection
          ->table('_ProjectsTree')
          ->where('id', $id)
          ->delete();
    }

    public function deleteLanguage($projectId)
    {
      return $this->webcatiBaseConnection
                  ->table('_LanguageProjects')
                  ->where('id_project', $projectId)
                  ->delete();
    }

    public function insertSystemDefaultQuestions($name, $questions)
    {
            return $this
                      ->webcatiConnection
                      ->table($name)
                      ->insertGetId($questions);
    }

    public function insertSystemDefaultLanguage($name, $texts)
    {
        $languageTable = $name . '_language';

        $this
          ->webcatiConnection
          ->table($languageTable)
          ->insert($texts);
    }

    public function getUnactiveProjects()
    {
        return $this
            ->webcatiBaseConnection
            ->table('_Projects')
            ->where('active', 1)
            ->get()
            ->all();
    }

    public function getStructureTable($projectId)
    {
        $projectName = $this->getProjectName($projectId);

        return $projectName . '_structure';
    }
}
