<?php

/**
 * Asterisk automatic dial class.
 * 
 * Public methods:
 * LogOn($Username, $Password, $ServerIP, $Port, $Extension, $interviewer)
 * LogOff()
 * Dial($SIP)
 * ExtensionStatus($SIP)
 *
 * Public properties:
 * DialStatus
 * -1 - Extension error or extension not available, check Extension Status return value
 * 0 - default, dial not initiated
 * 1 - external user pick up the phone
 * 2 - user busy
 * 5 - other telephone network problem
 * 8 - wrong, unallocated number
 * 9 - normal clearing
 * 30 - failed to create a socket
 * 31 - connect to socket failed
 * 32 - logon to socket failed
 * 33 - socket timeout on external line, no more data on the socket
 * 34 - packet timeout on external line, not my packet in 45s
 * 41 - connect to softphone failed, error on local extension
 * 44 - packet timeout on internal line
 * 
 * RM plus d.o.o.
 * version: 1.1.17 (2010.12.25)
 */
class RM_AsteriskManager {
	
	private $db;
	
	protected $FindTime; // store info when socket packet timeout will be executed 
	public $mSocket; // socket resource

	protected $WarpitUser; // loged in interviewer
	private $IntTelephoneNumber; // internal telephone number of loged in interviewer
	
	protected $AsteriskExtension; // local SIP extension
	
	protected $InternalChannel; // created channel to local SIP client
	protected $ExternalChannel; // created channel to external client
	
	protected $InternalID; // unique ID of local call
	protected $ExternalID; // unique ID of external call
	
	private $arrPackets=array(); // current complete packets, waiting to be used
	private $strLastBuffer; // last packet, possible uncomplete, waiting to be added to start to new read packet
	
	public $DialStatus=0; // store status of the last call
	
	
	/**
	 * Open Asterisk manager socket and connect to it
	 *
	 * @param int $SrvIP
	 * @param int $SrvPort
	 * @return bool
	 */
	protected function Connect($SrvIP,$SrvPort){
		set_time_limit(0);
		ob_implicit_flush(); // turn on automatic flush efter every print or echo
		$this->mSocket = socket_create(AF_INET, SOCK_STREAM, SOL_TCP);
		if ($this->mSocket < 0) {
			$this->DialStatus = 30;  // failed to create a socket
			return false;
		}
		if((@socket_connect($this->mSocket, $SrvIP, $SrvPort))) {
			$this->AM_LogWrite("Socket Connect() success!\r\n");
			return true;
		} else {
			$this->AM_LogWrite("--- LogON error -----\r\n".$this->GetSocketError($this->mSocket)."\r\n-----END LogON----\r\n\r\n");
			return false;
		}
	}
	

	/**
	 * Generate ActionID with addon string, if needed
	 *
	 * @param string $strAdd
	 * @return string Generated ActionID
	 */
	protected function GetActionID() {
		$strAdd = md5((microtime().rand()));
		return $strAdd;
	}
	
	
	/**
	 * Log ON to Asterisk Manager on specific port
	 *
	 * @param string $Username
	 * @param string $Password
	 * @param string $ServerIP
	 * @param int $Port
	 * @param string $actionID
	 * @return bool
	 */
	public function LogOn($Username, $Password, $ServerIP, $Port, $Extension, $interviewer){
		$actionID = $this->GetActionID();
		$this->AsteriskExtension = $Extension;
		
		$this->WarpitUser = $interviewer;
		$this->AM_LogWrite("--- WarpitUser -----\r\n".$this->WarpitUser."\r\n----------END WarpitUser---------\r\n\r\n");
		
		// prepare the login sequence
		$arrLoginPairs = array("Action" => "login", 
							   "Username" => $Username, 
							   "Secret" => $Password, 
							   "Events" => "on", 
							   "ActionID" => $actionID);
		$strAddPairs = $this->AddPair_AM($arrLoginPairs);
		
		// write Asterisk command pair to LOG file
		$this->AM_LogWrite("--- LogON 1. log -----\r\n".$strAddPairs."----------END 1.---------\r\n\r\n");

		// if, for whatever reaason socket exist, I close the socket before open it again and send the data to it
		if ($this->mSocket) {
			$this->Logoff();
		}
		
		// connect to Asterisk Manager socket
		if($this->Connect($ServerIP,$Port) == true){
			//  write to socket login string
			if ($this->SendPacket($strAddPairs) == true) {
				// check if login was success or failure
				$this->FindTime = $this->TimeNow(5); // set the timer for searching for my packet
				if ($this->GetLoginStatus($actionID)== true) {
					return true;
				} else {
					return false;					
				}
			} else {
				return false;
			}
		}else{
			$this->DialStatus = 31;  // connect to socket failed
			return false;
		}
	}


	/**
	 * Setting the socket timeout if no data on read 
	 *
	 * @param int $s
	 * @param int $ms
	 * @return unknown
	 */
	private function SetSocketRcvTimeout($s, $ms) {
		$timeout = array("sec"=>$s, "usec"=>$ms);
		if (socket_set_option($this->mSocket, SOL_SOCKET, SO_RCVTIMEO, $timeout)) {
			return true;
		} else {
			return false;
		}
	}
	

	/**
	 * Check for login status on socket
	 *
	 * @param string $actionID
	 * @return bool
	 */
	private function GetLoginStatus($actionID) {
		$MyPacket = array();
				
		if (($MyPacket = $this->GetResponse("ActionID", array($actionID)))) {
			if ($MyPacket['Response'] == "Success") {
				$this->AM_LogWrite("I got the packet, response Success!\r\n\r\n");
				return true;
			} else {
				$this->AM_LogWrite("I got the packet, response is failure!\r\n\r\n");
				$this->DialStatus = 32;  // logon to socket failure
				return false;
			}
		} else {
			$this->AM_LogWrite("Failed to get my login packets from the socket!\r\n\r\n");
			return false;
		}
	}
	
	
	/**
	 * Getting the new packets from $arrPackets
	 * When there is no more packets request to read from socket is made
	 * Transfer the packet back, if find the correct one
	 *
	 * @param string $ResponseKey
	 * @param array $ResponseItem
	 * @return array $MyPacket or false on failure
	 */
	protected function GetResponse($ResponseKey, $ResponseItem) {
		$OurPacket = false;
		$TimePass = false;
		$NewPacket = true;
		
		while($OurPacket == false AND $TimePass == false AND $NewPacket == true) {
			
			// if $arrPackets is empty I must fill it first with some packets
			if (!$this->arrPackets) {
				if ($this->GetNewPackets() == false) {
					$this->AM_LogWrite("Failed to get packets from the socket!\r\n");
					$NewPacket = false; 
				}
			}
			
			$OnePacket = array_shift($this->arrPackets);
			$this->AM_LogWrite("-".getmypid()."- OnePacket ---\r\n".$OnePacket."\r\n--- END OnePacket ---\r\n");
			
			$MyPacket = $this->ExplodePacket($OnePacket);

			// Check if this is my packet
			$this->AM_LogWrite("-".getmypid()."- Search for key/item ---\r\nFindKey: ".$ResponseKey."\r\nKey: ".implode(";", $ResponseItem)."\r\n--- END Search ---\r\n\r\n");
			
			if (in_array($MyPacket[$ResponseKey], $ResponseItem)) {
				$this->AM_LogWrite("-".getmypid()."- I found my packet! - $ResponseKey - ".implode(";", $ResponseItem)."\r\n");
				$OurPacket = true;
			}
			
			if ($this->FindTime < $this->TimeNow(0)) {
				$TimePass = true;
				$this->AM_LogWrite("-".getmypid()."- TimeCompare inside GetResponse()-\r\nTime limit: ".$this->FindTime." - Time now:".$this->TimeNow(0)."\r\n--- END TimeCompare ---\r\n\r\n");
			}
		}
		
		if ($NewPacket == false) {
			$this->AM_LogWrite(" -".getmypid()."- Failed to get packets from the socket - socket TIMEOUT!\r\n");
			$this->DialStatus = 33;  // socket timeout, no more data on the socket
			return false;
		} elseif ($TimePass == true AND $OurPacket == false) {
			$this->AM_LogWrite(" -".getmypid()."- Failed to get packets from the socket - no right packet TIMEOUT!\r\n");
			$this->DialStatus = 34;  // packet timeout, not my packet in 45s
			return false;
		} else {
			return $MyPacket;
		}	
	}
	
	
	/**
	 * Reading the packet from socket
	 * Merging the data until whole packet is received
	 * Then packet(s) are stored into $arrPackets
	 * Socket timout is set to 60s
	 *
	 * @return true when succes or false on error
	 */
	private function GetNewPackets() {
		unset($bufferPacket);
		unset($buffer);
		unset($ExplodedPackets);
		$this->AM_LogWrite("-".getmypid()."- No available packets, get new packets... ----\r\n");
		
		$cnt=0;
		$WholePacket = false;
		
		if ($this->SetSocketRcvTimeout(60,0)) 
			$this->AM_LogWrite("-".getmypid()."- socket timeout set to 60s: ----\r\n");
				
		$bufferPacket = $this->strLastBuffer; // add last item from uncompleeted array 
		while ($WholePacket == false) {
			if (($buffer = socket_read($this->mSocket, 2048))) {
				//$this->AM_LogWrite("----- counter: ".$cnt." <> buffer: --------\r\n".$buffer."------- END counter --------\r\n\r\n");
				$bufferPacket .= $buffer;
				//$this->AM_LogWrite("----- counter: ".$cnt." <> bufferPacket: --------\r\n".$bufferPacket."------- END counter --------\r\n\r\n");
				if (strpos($bufferPacket,"\r\n\r\n") !== false) {
					$ExplodedPackets = explode("\r\n\r\n", $bufferPacket);
					$this->strLastBuffer = array_pop($ExplodedPackets);
					$this->arrPackets = $ExplodedPackets;
					$WholePacket = true;
					//$this->AM_LogWrite("\r\n--".getmypid()."-- I read the whole packet -----\r\n".$bufferPacket."\r\n--- end whole packet...\r\n");
				}	
			} else {
				$this->AM_LogWrite("-".getmypid()."- ERROR ReadMyPAcket ----\r\n".$this->GetSocketError($this->mSocket)."\r\n---- END error ----\r\n");
				return false;				
			}
			$cnt++;
			usleep(100);
		}
		return true;
	}
		
	
	/**
	 * Dial the number
	 * First try the internal SIP client, if true
	 * Try the external line and get the status of the call
	 *
	 * @param string $SIP
	 * @return true when succes or false on error
	 */
	public function Dial($SIP, $Context, $CallerID, $Number2Call) {
		$actionID = $this->GetActionID();
		$this->IntTelephoneNumber = $SIP;
		//$Number2Call = "031689240"; // used for testing, do not uncomment, if you dont know what you are doing
		
		$arrLoginPairs = array("Action" => "Originate", 
								"Channel" => "SIP/".$SIP,
								"Context" => $Context, 
								"Exten" => $Number2Call,
								"Priority" => "1",
								"Callerid" => $CallerID, 
								"Timeout" => "5000",
								"ActionID" => $actionID);
		
		$strAddPairs = $this->AddPair_AM($arrLoginPairs);
		
		// write Asterisk command pair to LOG file
		$this->AM_LogWrite("----- Dial ------\r\n".$strAddPairs."----- END Dial ------\r\n\r\n");
		
		if ($this->SendPacket($strAddPairs) == true) {
			// check if login was success or failure
			$this->FindTime = $this->TimeNow(5); // set the timer for searching for my packet
			if ($this->GetIntDialStatus($actionID) == true) {
				$this->AM_LogWrite("----- Dial internal ------\r\n Dial internal success \r\n----- END Dial internal ------\r\n\r\n");
				$this->FindTime = $this->TimeNow(45); // set the timer for searching for my packet
				if ($this->GetExtChannel() == true) {
					$this->AM_LogWrite("----- Dial external ------\r\n Dial external success \r\n----- END Dial external ------\r\n\r\n");
					$this->FindTime = $this->TimeNow(45); // set the timer for searching for my packet
					if ($this->GetExtStatus() == true) {
						$this->AM_LogWrite("----- Dial status ------\r\n I got the status \r\n----- END Dial status ------\r\n\r\n");
						return true;
					} else {
						$this->AM_LogWrite("----- Dial status ------\r\n Return false! \r\n----- END Dial status ------\r\n\r\n");
					}
				}
			}
			return false;
		} else {
			return false;
		}
	}
	
	
	public function Hangup($SIP) {
		$arrLoginPairs = array("Action" => "Hangup", 
								"Channel" => $this->GetChannel($SIP)
							   );
		
		$strAddPairs = $this->AddPair_AM($arrLoginPairs);
		
		// write Hangup command to LOG file
		$this->AM_LogWrite("----- Hangup ------\r\n".$strAddPairs."----- END Hangup ------\r\n\r\n");
		
		if ($this->SendPacket($strAddPairs) == true) {
			return true;
		} else {
			return false;
		}
	}
	
	

	/**
	 * Check the internal line and get the internal line status
	 *
	 * @param unknown_type $SIP
	 * @return int LineStatus
	 * 
	 * -9 = timeout, error getting any response from Asterisk
	 * -8 = Response error, ExtensionState response is error
	 * -7 = Unable to send data to socket
	 * -1 = Extension not found 
	 *  0 = Idle 
	 *  1 = In Use 
	 *  2 = Busy 
	 *  4 = Unavailable 
	 *  8 = Ringing 
	 *  16 = On Hold
	 */
	public function ExtensionStatus($SIP, $Context) {
		$LineStatus = -1; // set the line status to non existing extension
		$actionID = $this->GetActionID();
		
		$arrLoginPairs = array("Action" => "ExtensionState", 
								"Context" => $Context,
								"Exten" => $SIP,
								"ActionID" => $actionID
							   );
		
		$strAddPairs = $this->AddPair_AM($arrLoginPairs);
		
		// write Hangup command to LOG file
		$this->AM_LogWrite("----- ExtensionStatus ------\r\n".$strAddPairs."----- END ExtensionStatus ------\r\n\r\n");
		
		if ($this->SendPacket($strAddPairs) == true) {
			// get the line status
			$this->FindTime = $this->TimeNow(5); // set the timer for searching for my packet
			$LineStatus = $this->GetIntLineStatus($actionID);
			$this->AM_LogWrite("----- LineStatus ------\r\n Line status = ".$LineStatus."\r\n----- END LineStatus ------\r\n\r\n");
			$this->DialStatus = -1;
			return $LineStatus;
		} else {
			$this->DialStatus = -1;
			return -7; // unable to write to socket
		}
	}
	
	
	/**
	 * Checks if the internal SIP client was succesfully contacted 
	 *
	 * @param string $actionID
	 * @return true when succes or false on error
	 */
	private function GetIntDialStatus($actionID) {
		$MyPacket = array();
				
		if (($MyPacket = $this->GetResponse("ActionID", array($actionID)))) {
			if ($MyPacket['Response'] == "Success") {
				return true;
			} else {
				$this->DialStatus = 41;
				return false;
			}
		} else {
			$this->DialStatus = 44;
			return false;
		}
	}

	
	private function GetIntLineStatus($actionID) {
		$MyPacket = array();
				
		if (($MyPacket = $this->GetResponse("ActionID", array($actionID)))) {
			if ($MyPacket['Response'] == "Success") {
				return $MyPacket['Status'];
			} else {
				$this->DialStatus = -1;
				return -8; // error when sending packet for line status
			}
		} else {
			return -9; // error getting response for line status
		}
	}

	
	/**
	 * Get the local channel and store it to $InternalChannel
	 * if checked packet is not ours, checked recursively next ones
	 * until timeout or get our packet
	 *
	 * @return true when succes or false on error
	 */
	private function GetExtChannel() {
		$MyPacket = array();
				
		if (($MyPacket = $this->GetResponse("Event", array("Newchannel")))) {
			$array_MyPacket = explode("-", $MyPacket['Channel'], 2);
			$this->AM_LogWrite("I got GetExtDialStatus:\r\n".$array_MyPacket[0]."\r\n----- END GetExtDialStatus -----\r\n\r\n");
			if ($array_MyPacket[0] == "SIP/".$this->AsteriskExtension) {
				$this->InternalChannel = $MyPacket['Channel'];
				$this->InternalID = $MyPacket['Uniqueid'];
				$this->SaveChannel();
				$this->AM_LogWrite("--- Save channel ---\r\n".$MyPacket['Channel']." - ".$MyPacket['Uniqueid']."\r\n--- END Save ---\r\n\r\n");
				return true;
			} else {
				// if we did not find the correct packet, we search forward
				$this->GetExtChannel();
				return false;
			}
		} else {
			return false;
		}
	}
	
	
	/**
	 * Checking the status of the call
	 * first waiting for correct packet, if found checks the status
	 *
	 * Main hangup causes:
		1 unallocated number 
		2 no route to network 
		3 no route to destination 
		16 normal call clearing 
		17 user busy 
		18 no user responding 
		19 no answer from the user 
		20 subscriber absent 
		21 call rejected 
		22 number changed (w/o diagnostic) 
		22 number changed (w/ diagnostic) 
		23 redirection to new destination 
		26 non-selected user clearing 
		27 destination out of order 
		28 address incomplete 
		29 facility rejected 
		31 normal unspecified 
	 * 
	 * @return true when succes or false on error
	 */
	private function GetExtStatus() {
		$MyPacket = array();
				
		if (($MyPacket = $this->GetResponse("Event", array("Link","Bridge","Hangup")))) {
			switch ($MyPacket['Event']) {
				case "Link": 
					$this->AM_LogWrite("--- Linked value ---\r\nHangupChannel: ".$MyPacket['Channel1']."\r\nSavedChannel: ".$this->InternalChannel."\r\n--- END Linked value ---\r\n\r\n");
					if ($MyPacket['Channel1'] == $this->InternalChannel) {
						$this->AM_LogWrite("--- Checking status ---\r\nLinked: ".$MyPacket['Channel1']."\r\n--- END Check ---\r\n\r\n");
						$this->DialStatus = 1;  // external user pick up the phone, lines linked
					} else {
						$this->GetExtStatus(); // if we did not find the correct packet, we search forward
					}
					break;
				case "Bridge": 
					$this->AM_LogWrite("--- Bridged value ---\r\nHangupChannel: ".$MyPacket['Channel1']."\r\nSavedChannel: ".$this->InternalChannel."\r\n--- END Bridged value ---\r\n\r\n");
					if ($MyPacket['Channel1'] == $this->InternalChannel) {
						$this->AM_LogWrite("--- Checking status ---\r\nBridged: ".$MyPacket['Channel1']."\r\n--- END Check ---\r\n\r\n");
						$this->DialStatus = 1;  // external user pick up the phone, lines bridged
					} else {
						$this->GetExtStatus(); // if we did not find the correct packet, we search forward
					}
					break;
				case "Hangup": 
					$this->AM_LogWrite("--- Hangup value ---\r\nHangupID: ".$MyPacket['Uniqueid']."\r\nSavedID: ".$this->InternalID."\r\n--- END Hangup value ---\r\n\r\n");
					if ($MyPacket['Uniqueid'] == $this->InternalID) {
						$this->AM_LogWrite("--- Checking status ---\r\nHangup: ".$MyPacket['Cause-txt']." - [".$MyPacket['Cause']."]\r\n--- END Check ---\r\n\r\n");
						if ($MyPacket['Cause'] == 1) $this->DialStatus = 8;  // wrong, unallocated number
						if ($MyPacket['Cause'] == 16 OR $MyPacket['Cause'] == 0) $this->DialStatus = 9;  // normal clearing
						if ($MyPacket['Cause'] == 17) $this->DialStatus = 2;  // user busy
						if ($MyPacket['Cause'] >= 2 AND $MyPacket['Cause'] <= 15) $this->DialStatus = 5;  // other telephone network problem
						if ($MyPacket['Cause'] >= 18) $this->DialStatus = 5;  // other telephone network problem
					} else {
						$this->GetExtStatus(); // if we did not find the correct packet, we search forward
					}
					break;
			}
			return true;
		} else {
			return false;
		}
	}
	
	
	
	/**
	 * Make array out of Asterisk Packet -> command into $key and action into #item
	 *
	 * @param string $packet
	 * @return array
	 */
	private function ExplodePacket($packet) {
		unset($arrMyPacket);
		
		$packet = trim($packet);
		$arrPacket = explode("\r\n", $packet);
		foreach ($arrPacket AS $PacketLine) {
			$arrPacketLine = explode(":",$PacketLine);
			$arrMyPacket[trim($arrPacketLine[0])] = trim($arrPacketLine[1]);
		}
		return $arrMyPacket;
	}
	

	
	/**
	 * Get last socket status, if resource is given get status of specific socket
	 *
	 * @param recource $res
	 * @return string
	 */
	private function GetSocketError($res = null) {
		return socket_strerror(socket_last_error($res));
	}

	
	
	/**
	 * Send string to write to socket
	 *
	 * @param string $buffer
	 * @return bool
	 */
	protected function SendPacket($buffer){
		if (socket_write($this->mSocket, $buffer, strlen($buffer)) === false) {
			$this->AM_LogWrite("--- ERROR SendPacket ---\r\n".$this->GetSocketError($this->mSocket)."\r\n--- END SendPacket ---\r\n\r\n");
			return false;
		} else {
			return true;
		}
	}
	
        
	/**
	 * Close socket to Asterisk manager
	 *
	 * @return void
	 */        
	public function LogOff(){
       	socket_close($this->mSocket);
       	unset($this->mSocket);
	}


	/**
	 * Create Asterisk Manager command pairs out of array
	 *
	 * @param array $arrPairs
	 * @return string
	 */
	protected function AddPair_AM($arrPairs) {
		unset($tmp);
		
		foreach ($arrPairs AS $key => $item) {
			$tmp .= $key.": ".$item."\r\n";
		}
		$tmp .= "\r\n";
		return $tmp;
	}

	
	/**
	 * Return curent unix time stamp, added with number of seconds from parameter
	 *
	 * @param int $t
	 * @return int Unix time stamp
	 */
	protected function TimeNow($t) {
		return time()+$t;
	}
	
	
	
	private function SaveChannel() {
		global $SQLconnect;
		global $CATI_base_structure;
		
		$sql = "REPLACE _Asterisk_RMconnect SET sip='".$this->IntTelephoneNumber."', id_user='".$this->WarpitUser."', 
						channel_internal='".$this->InternalChannel."', id_internal='".$this->InternalID."', connect=NULL";
		$this->AM_LogWrite("--- SQL save channel ---\r\n".$sql."\r\n--- END save channel ---\r\n\r\n");
		mysql_db_query($CATI_base_structure, $sql , $SQLconnect);
	}
	
	
	
	private function GetChannel($SIP) {
		global $SQLconnect;
		global $CATI_base_structure;
		
		$sql = "SELECT channel_internal FROM _Asterisk_RMconnect WHERE sip='".$SIP."'";
		$this->AM_LogWrite("--- SQL get channel ---\r\n".$sql."\r\n--- END get channel ---\r\n\r\n");
		$result = mysql_db_query($CATI_base_structure, $sql , $SQLconnect);
		$row = mysql_fetch_array($result);
		return $row['channel_internal'];
	}
	
	
	
	/**
	 * Prepare the log location
	 *
	 * @return string
	 */
	private function LogLocation() {
		return "/tmp/RM_AsteriskManager___".$this->AsteriskExtension.".log";
	}
	

	/**
	 * set the time zone and return the date
	 *
	 * @return date in the "H:i:s:u [j.n.Y]" format
	 */
	protected function GetTimeLog() {
		return date("H:i:s:u [j.n.Y]");
	}
	

	/**
	 * Write debug info to a log file
	 *
	 * @param string $strLog
	 */
	private function AM_LogWrite($strLog) {
		$strLog = $this->GetTimeLog()." - ".$this->AsteriskExtension."[".$this->WarpitUser."] - ".$strLog;
		
		if (file_put_contents($this->LogLocation(), $strLog, FILE_APPEND)) {
			return true;
		} else {
			return false;
		}
	}
	
	
}

?>