<?php
class ReportGeneratorBase
{
    protected $mainCategory;
    protected $ultimativeCalculator;
    protected $calculator;
    protected $columns;
    protected $decimalPlaces;
    protected $reports;
    protected $html;
    protected $exportData;
    protected $tableTitle;
    protected $languagePack;
    protected $languageLabels;

    public function __construct($calculator, $ultimativeCalculator, $languagePack, $tableTitle)
    {
        $this->html = "";
        $this->calculator = $calculator;
        $this->mainCategory = $calculator->getStructure();
        $this->ultimativeCalculator = $ultimativeCalculator;
        $this->decimalPlaces = 2;
        $this->exportData = Array();
        $this->tableTitle = $tableTitle;
        $this->languagePack = $languagePack;
        $this->languageLabels = Array();
    }

    public function getDataForExport()
    {
        return $this->exportData;
    }

    public function getColumnsHeaders()
    {
        return $this->columns;
    }

    public function addReportGenerator($reportGenerator)
    {
        $this->reports[] = $reportGenerator;
    }

    public function getGenerateHtml($onlyTable = FALSE)
    {
        $this->generateHtml();
        return $this->html;
    }

    protected function generateHtml($onlyTable = FALSE)
    {
        if(!$onlyTable)
        {
            $this->html = "<html>";
            $this->html .= $this->headerHtml();
            $this->html .= "<body>";
        }

        if(isset($this->tableTitle))
        {
            $this->html .= "<h1>" . $this->tableTitle . "</h1>";
        }

        $this->html .= "<table>";
        $this->html .= $this->tableHeaderHtml();
        $this->html .= $this->tableBodyHtml();
        $this->html .= $this->tableFooterHtml();
        $this->html .= "</table>";

        if(!is_null($this->reports))
        {
            $this->html .= "<br><br><br>";
            foreach ($this->reports as $reportGenerator)
            {
                $this->html .= $reportGenerator->getGenerateHtml(TRUE);
            }
        }

        if(!$onlyTable)
        {
            $this->html .= "</body></html>";
        }

    }

}





















class ReportGenerator extends ReportGeneratorBase
{
    protected $specialColumns;

    public function __construct($calculator, $ultimativeCalculator, $languagePack, $specialColumns)
    {
        parent::__construct($calculator, $ultimativeCalculator, $languagePack);
        $this->specialColumns = $specialColumns;

        $this->loadLanguageLabels();
        $this->createHeaders();
    }

    private function loadLanguageLabels()
    {
        $lp = $this->languagePack;
        $this->languageLabels["address"]             =   $lp->GetLabel("special-report" , "header-address" , "Address");
        $this->languageLabels["ultimative-score"]    =   $lp->GetLabel("special-report" , "header-ultimative-score" , "Ultimative score");
        $this->languageLabels["visit-score-procent"] =   $lp->GetLabel("special-report" , "header-visit-score-procent" , "Visit score %");
        $this->languageLabels["category-points"]     =   $lp->GetLabel("special-report" , "header-category-points" , "Category points - ");
        $this->languageLabels["category-procent"]    =   $lp->GetLabel("special-report" , "header-category-procent" , "Category % - ");
        $this->languageLabels["question-points"]     =   $lp->GetLabel("special-report" , "header-question-points" , "Question points - ");
        $this->languageLabels["question-procent"]    =   $lp->GetLabel("special-report" , "header-question-procent" , "Question % -");
    }

    protected function createHeaders()
    {
        $this->columns[] = $this->languageLabels["address"];
        $this->columns[] = $this->languageLabels["ultimative-score"];
        $this->columns[] = $this->languageLabels["visit-score-procent"];
        $this->addHeadersRecursive($this->mainCategory);
    }

    protected function addHeadersRecursive($category)
    {
        if($category->showInStatisticTableRecursive() && !$category->isRoot())
        {
            $this->columns[] = $this->languageLabels["category-points"] . $category->getName();
            $this->columns[] = $this->languageLabels["category-procent"] . $category->getName();
            $this->addQuestionHeaders($category);
        }

        if($category->hasCategories())
        {
            foreach ($category->getCategories() as $subCategory)
            {
                $this->addHeadersRecursive($subCategory);
            }
        }
    }

    protected function addQuestionHeaders($category)
    {
        if($category->hasQuestions())
        {
            foreach ($category->getQuestions() as $question)
            {
                $this->columns[] = $this->languageLabels["question-points"]  . $question->getName();
                $this->columns[] = $this->languageLabels["question-procent"] . $question->getName();
            }
        }
    }

    protected function headerHtml()
    {
        return "
        <head>
            <style>
            table {
                border: 1px solid black;
                width: 100%;
                 display: table-cell;
            }
            td, th{
                border: 1px solid black;
                padding: 10px;
                text-align: center;
                 display: table-cell;
            }

            .tableColumn {
                width: 200px;
                text-align: center;
            }
            .categoryColumn
            {
                background-color: #daffcb;
            }
            .ultimateiveColumn
            {
                background-color: #ffb79e;
            }
            </style>
        </head>
        ";
    }

    protected function tableHeaderHtml()
    {
        $hHtml = "<tr>";
        foreach ($this->columns as $column)
        {
            $hHtml .= "<th><div class='tableColumn'>$column</div></th>";
        }
        $hHtml .= "</tr>";
        return $hHtml;
    }

    protected function tableFooterHtml()
    {
        return "";
    }

    protected function tableBodyHtml()
    {
        $tbodyHtml = "";
        foreach ($this->calculator->getRecordsResult() as $id => $record)
        {
            $tbodyHtml .= $this->tableRowHtml($this->mainCategory, $id, $record);
        }
        return $tbodyHtml;
    }

    private function tableRowHtml($category, $recordId, $record)
    {
        $rHtml = "<tr>";
        $specialText = $this->specialColumns[$recordId];
        $rHtml .= "<td >" . $specialText . "</td>";

        $rHtml .= "<td class = 'ultimateiveColumn'>" .  number_format($this->ultimativeCalculator->ultimativeScore(), $this->decimalPlaces) . "</td>";
        $rHtml .= "<td class = 'categoryColumn'>" . number_format($this->calculator->recordTotalProcent($record), $this->decimalPlaces) . "</td>";


        $rHtml .= $this->tableRowHtmlRecursive($category, $record);

        $rHtml .= "</tr>";
        return $rHtml;
    }

    private function tableRowHtmlRecursive($category, $record)
    {
        $calc = $this->calculator;
        $rHtml = "";
        if($category->showInStatisticTableRecursive() && !$category->isRoot())
        {
            $categoryValue = $calc->categoryRecordTotal($category->getId(), $record);
            $rHtml .= "<td class='categoryColumn'>" . $categoryValue . "</td>";

            $categoryValue = number_format( $calc->categoryRecordTotalProcent($category->getId(), $record), 2);

            $rHtml .= "<td class='categoryColumn'>" . $categoryValue . "</td>";
            if($category->hasQuestions())
            {
                foreach ($category->getQuestions() as $question)
                {
                    $questionValue = $calc->questionRecordTotal($question, $record);
                    $rHtml .= "<td>" . $questionValue . "</td>";

                    $questionValue = number_format( $calc->questionRecordTotalProcent($question, $record), 2);
                    $rHtml .= "<td>" . $questionValue . "</td>";
                }
            }

        }

        if($category->hasCategories())
        {
            foreach ($category->getCategories() as $subcategory)
            {
                $rHtml .= $this->tableRowHtmlRecursive($subcategory, $record);
            }
        }
        return $rHtml;
    }


}




















class ReportGeneratorGroups extends ReportGeneratorBase
{
    protected $groups;
    protected $rowCount;

    public function __construct($calculator, $ultimativeCalculator, $languagePack)
    {
        parent::__construct($calculator, $ultimativeCalculator, $languagePack);
        $this->groups = $calculator->getGroups();
        $this->loadLanguageLabels();

        $this->createHeaders();
    }

    private function loadLanguageLabels()
    {
        $lp = $this->languagePack;
        $this->languageLabels["average-total"]  =   $lp->GetLabel("special-report-group","header-average-total","Avg. Total");
        $this->languageLabels["visit-total"]    =   $lp->GetLabel("special-report-group","header-visit-total","Visit total");
    }

    protected function createHeaders()
    {
        $this->columns[] = $this->languageLabels["average-total"] . " " . number_format( $this->ultimativeCalculator->ultimativeScore(), $this->decimalPlaces) . "%";
        $this->addOtherHeaders();
    }

    protected function addOtherHeaders()
    {
        foreach ($this->groups as $groupId => $group)
        {
            $this->columns[] =  $this->languageLabels["visit-total"] . " " . $group["label"] . ": " . number_format( $this->calculator->groupScore($groupId), $this->decimalPlaces) . "%";
        }
    }

    protected function headerHtml()
    {
        return "
        <head>
            <style>
            table
            {
                border-collapse: collapse;
            }
            td, th
            {
                padding: 5px;
                text-align: center;
            }

            .oddRow
            {
                background-color: #ffffff;
            }

            .evenRow
            {
                background-color: #f2f2f2;
            }

            .categoryRow
            {
                background-color: #c2c2c2;
            }

            .headerRow
            {
                background-color: #215968;
                color: white;
                padding: 20px;
            }

            .bold
            {
                font-weight: 700;
            }

            .textAlignLeft
            {
                text-align: left;
            }

            </style>
        </head>
        ";
    }

    protected function tableHeaderHtml()
    {
        $hHtml = "<tr>";
        foreach ($this->columns as $column)
        {
            $hHtml .= "<th class='headerRow'>" . $column . "</th>";
        }
        $hHtml .= "</tr>";
        return $hHtml;
    }

    protected function tableFooterHtml()
    {
        return "";
    }

    protected function tableBodyHtml()
    {
        $tbodyHtml = "";

        $tbodyHtml .= $this->tableRowHtmlRecursive($this->mainCategory);

        return $tbodyHtml;
    }

    protected function tableRowHtmlRecursive($category, $counter = 0)
    {
        $htmlRows = "";

        if($category->showInStatisticTableRecursive() && !$category->isRoot())
        {
            $htmlRows .= $this->tableRowCategoryHtml($category);
        }

        if($category->hasCategories())
        {
            foreach ($category->getCategories() as $subcategory)
            {
                $htmlRows .= $this->tableRowHtmlRecursive($subcategory, $counter);
            }
        }

        if($category->hasQuestions() && $category->showInStatisticTableRecursive())
        {
            foreach ($category->getQuestions() as $question)
            {
                $htmlRows .= $this->tableRowQuestionHtml($question);
            }
        }

        return $htmlRows;
    }


    protected function tableRowCategoryHtml($category)
    {
        $this->rowCount++;
        $cssClass = "categoryRow";
        $cssClass .= " bold";

        $rHtml = "<tr >";
        $rHtml .= "<td class='$cssClass '>" . $category->getName() . "</td>";

        foreach ($this->groups as $groupId => $group) {
            $rHtml .= "<td class='$cssClass'>" . number_format( $this->calculator->categroyGroupScore($category->getId(), $groupId), $this->decimalPlaces ) . "%</td>";
        }

        $rHtml .= "</tr>";
        return $rHtml;
    }

    protected function tableRowQuestionHtml($question)
    {
        $this->rowCount++;
        $cssClass = ($this->rowCount % 2 == 0) ? "evenRow" : "oddRow";

        $rHtml = "<tr >";
        $rHtml .= "<td class='$cssClass textAlignLeft'>" . $question->getLabel() . "</td>";

        foreach ($this->groups as $groupId => $group)
        {
            $rHtml .= "<td class='$cssClass'>" . number_format( $this->calculator->questionGroupScore($question, $groupId), $this->decimalPlaces ) . "%</td>";
        }

        $rHtml .= "</tr>";
        return $rHtml;
    }
}



















class ReportGeneratorGroupsReview extends ReportGeneratorBase
{
    protected $groupHeadersCount;
    protected $groups;

    public function __construct($calculator, $ultimativeCalculator, $languagePack)
    {
        parent::__construct($calculator, $ultimativeCalculator, $languagePack);
        $this->groups = $calculator->getGroups();
        $this->loadLanguageLabels();

        $this->createHeaders();
    }

    private function loadLanguageLabels()
    {
        $lp = $this->languagePack;
        $this->languageLabels["average-total"]  =   $lp->GetLabel("special-report-group","header-average-total","Avg. Total");
        $this->languageLabels["address"]        =   $lp->GetLabel("special-report-group","header-address","Address");
        $this->languageLabels["visits-total"]   =   $lp->GetLabel("special-report-group","header-visits-total","Visits total");
        $this->languageLabels["visit-prefix"]   =   $lp->GetLabel("special-report-group","header-visit-prefix","Visit");
    }

    protected function createHeaders()
    {
        $this->columns[] = $this->languageLabels["address"];
        $this->columns[] = $this->languageLabels["average-total"];
        $this->columns[] = $this->languageLabels["visits-total"];
        $this->addOtherHeaders();
    }

    protected function addOtherHeaders()
    {
        $this->groupHeadersCount = 0;
        $groups = $this->groups;
        foreach ($groups as $group)
        {
            $groupSize = count($group["ids"]);
            if($groupSize > $this->groupHeadersCount)
            {
                $this->groupHeadersCount = $groupSize;
            }
        }

        for($i = 0; $i < $this->groupHeadersCount; $i++)
        {
            $this->columns[] = $this->languageLabels["visit-prefix"] . " " . ($i+1);
        }

    }

    protected function headerHtml()
    {
        return "
        <head>
            <style>
            table {
                border-collapse: collapse;
            }
            td, th{

                padding: 5px;
                text-align: center;
            }

            .oddRow
            {
                background-color: #ffffff;
            }

            .evenRow
            {
                background-color: #f2f2f2;
            }

            .categoryRow
            {
                background-color: #c2c2c2;
            }

            .headerRow
            {
                background-color: #215968;
                color: white;
                padding: 20px;
            }

            .bold
            {
                font-weight: 700;
            }

            .textAlignLeft
            {
                text-align: left;
            }
            </style>
        </head>
        ";
    }

    protected function tableHeaderHtml()
    {
        $hHtml = "<tr>";
        foreach ($this->columns as $column)
        {
            $hHtml .= "<th class='headerRow'>" . $column . "</th>";
        }
        $hHtml .= "</tr>";
        return $hHtml;
    }

    protected function tableFooterHtml()
    {
        return "";
    }

    protected function tableBodyHtml()
    {
        $tbodyHtml = "";
        $evenCounter = 0;
        foreach ($this->groups as $groupId => $group)
        {
            $tbodyHtml .= $this->tableRowHtml($groupId, ($evenCounter++ % 2 == 0));
        }
        return $tbodyHtml;
    }

    protected function tableRowHtml($groupId, $isEven)
    {
        $group = $this->groups[$groupId];

        $records = $this->calculator->getRecordsResult();
        $cssClass = ($isEven) ? "evenRow" : "oddRow";

        $rHtml = "<tr >";
        $rHtml .= "<td class='$cssClass textAlignLeft'>" . $group["label"] . "</td>";
        $rHtml .= "<td class='$cssClass'>" . number_format( $this->ultimativeCalculator->ultimativeScore(), $this->decimalPlaces) . "%</td>";
        $rHtml .= "<td class='$cssClass'>" . number_format( $this->calculator->groupScore($groupId), $this->decimalPlaces) . "%</td>";

        $columnCounter = 0;
        foreach ($group["ids"] as $recordId) {
            $columnCounter++;
            $record = $records[$recordId];
            $rHtml .= "<td class='$cssClass'>" . number_format( $this->calculator->recordTotalProcent($record), $this->decimalPlaces ) . "%</td>";
        }

        //add empty dummy rows
        for($i = $columnCounter; $i < $this->groupHeadersCount; $i++)
        {
            $rHtml .= "<td class='$cssClass'>/</td>";
        }

        $rHtml .= "</tr>";
        return $rHtml;
    }
}

























class ReportGeneratorNegativePositive extends ReportGeneratorBase
{
    private $specialColumns;

    public function __construct($calculator, $ultimativeCalculator, $languagePack, $specialColumns, $tableTitle)
    {
        parent::__construct($calculator, $ultimativeCalculator, $languagePack, $tableTitle);
        $this->specialColumns = $specialColumns;
        $this->loadLanguageLabels();

        $this->createHeaders();
    }

    private function loadLanguageLabels()
    {
        $lp = $this->languagePack;
        $this->languageLabels["address"]  =   $lp->GetLabel("total-report-by-row","header-address","Address");
        $this->languageLabels["total"]    =   $lp->GetLabel("total-report-by-row","header-total","Total");
    }

    protected function createHeaders()
    {
        $this->columns[] = $this->languageLabels["address"];
        $this->columns[] = $this->languageLabels["total"];
        $this->addHeadersRecursive($this->mainCategory);
    }

    protected function addHeadersRecursive($category)
    {
        if($category->showInStatisticTableRecursive() && !$category->isRoot())
        {
            $this->columns[] = $category->getName();
            $this->addQuestionHeaders($category);
        }

        if($category->hasCategories())
        {
            foreach ($category->getCategories() as $subCategory)
            {
                $this->addHeadersRecursive($subCategory);
            }
        }
    }


    protected function addQuestionHeaders($category)
    {
        if($category->hasQuestions())
        {
            foreach ($category->getQuestions() as $question)
            {
                $this->columns[] = $question->getName();
            }
        }
    }

    protected function headerHtml()
    {
        return "
        <head>
            <style>
            table
            {
                border-collapse: collapse;
            }
            td, th
            {
                padding: 10px;
                text-align: center;
            }
            th
            {
                min-width: 150px;
                background-color: #215968;
                color: white;
            }

            .addressColumn{
                min-width: 520px;
            }

            .oddRow
            {
                background-color: #ffffff;
            }

            .evenRow
            {
                background-color: #D2D2D2;
            }

            .categoryColumn
            {
                background-color:  #a8a8a8;
            }
            </style>
        </head>
        ";
    }

    protected function tableHeaderHtml()
    {
        $addressColumn = TRUE;
        $hHtml = "<tr>";
        foreach ($this->columns as $column)
        {
            if($addressColumn)
            {
                $hHtml .= "<th class='addressColumn'><div class='headerRow'>$column</div></th>";
                $addressColumn = FALSE;
            }
            else
            {
                $hHtml .= "<th class='headerRow'><div class='headerRow'>$column</div></th>";
            }
        }
        $hHtml .= "</tr>";
        return $hHtml;
    }

    protected function tableFooterHtml()
    {
        return "";
    }

    protected function tableBodyHtml()
    {
        $tbodyHtml = "";

        $oddEvenCounter = 0;
        foreach ($this->calculator->getRecordsResult() as $id => $record)
        {
            $data = Array();
            $tbodyHtml .= $this->tableRowHtml($this->mainCategory, $id, $record, ($oddEvenCounter % 2), $data);
            $this->exportData[] = $data;
            $oddEvenCounter++;
        }
        return $tbodyHtml;
    }

    protected function tableRowHtml($category, $recordId, $record, $isEven = TRUE, &$dataForExport)
    {
        $class = ($isEven) ? "evenRow" : "oddRow";

        $rHtml = "<tr class='$class'>";
        $specialText = $this->specialColumns[$recordId];
        $dataForExport[] = $specialText;
        $rHtml .= "<td >" . $specialText . "</td>";

        $rHtml .= $this->tableRowHtmlRecursive($category, $record, $dataForExport);

        $rHtml .= "</tr>";
        return $rHtml;
    }

    protected function tableRowHtmlRecursive($category, $record, &$dataForExport)
    {
        $calc = $this->calculator;
        $rHtml = "";
        if($category->isRoot())
        {
            $totalValue = number_format(  $calc->categoryPositiveNegativeTotalProcent($category->getId(), $record, TRUE), $this->decimalPlaces);
            $dataForExport[] = $totalValue;
            $rHtml .= "<td class='categoryColumn'>" . $totalValue . "</td>";
        }
        if($category->showInStatisticTableRecursive() && !$category->isRoot())
        {
            $categoryValue = number_format(  $calc->categoryPositiveNegativeTotalProcent($category->getId(), $record, TRUE), $this->decimalPlaces);
            $dataForExport[] = $categoryValue;
            $rHtml .= "<td class='categoryColumn'>" . $categoryValue . "</td>";
            if($category->hasQuestions())
            {
                foreach ($category->getQuestions() as $question)
                {
                    $questionValue = number_format( $calc->questionPositiveNegativeTotalProcent($question, $record, TRUE), $this->decimalPlaces);

                    $dataForExport[] = $questionValue;
                    $rHtml .= "<td>" . $questionValue . "</td>";
                }
            }

        }

        if($category->hasCategories())
        {
            foreach ($category->getCategories() as $subcategory)
            {
                $rHtml .= $this->tableRowHtmlRecursive($subcategory, $record, $dataForExport);
            }
        }
        return $rHtml;
    }
}
?>
