<?php

namespace SolveX\Http;

use SolveX\Http\UploadedFile;
use SolveX\Interfaces\RequestInterface;
use SolveX\Exceptions\InvalidCsrfTokenException;

/**
 * Request class provides a number of methods for examining the HTTP
 * request.
 *
 */
class Request implements RequestInterface
{
    /**
     * HTTP headers received.
     */
    protected $headers = [];


    /**
     * $_GET, $_POST. Also works when HTTP method is PUT or DELETE.
     */
    protected $params = [];


    public function __construct()
    {
        $headers = function_exists('getallheaders') ? getallheaders() : [];
        $lowercased = [];
        foreach ($headers as $name => $value) {
            $lowercased[strtolower($name)] = $value;
        }

        $this->headers = $lowercased;

        $method = $_SERVER['REQUEST_METHOD'];

        // TODO: HTML form method override (PUT and DELETE are not natively supported)
        // http://docs.slimframework.com/routing/delete/

        if ($method === 'PUT' || $method === 'DELETE')
            parse_str(file_get_contents('php://input'), $this->params);
        else if ($method == 'GET')
            $this->params = $_GET;
        else if ($method == 'POST')
            $this->params = $_POST;
    }


    /**
     * Is this request AJAX? Checks for HTTP_X_REQUESTED_WITH header.
     *
     * @return bool
     */
    public function isAjax()
    {
        if (empty($_SERVER['HTTP_X_REQUESTED_WITH']))
            return false;

        if (strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) !== 'xmlhttprequest')
            return false;

        return true;
    }


    /**
     * Returns the HTTP request method (e.g. GET, POST, DELETE, PUT)
     *
     * @return string Request method.
     */
    public function method()
    {
        return $_SERVER['REQUEST_METHOD'];
    }


    /**
     * Returns the request path.
     *
     * Removes prefix if used in a subfolder. That is, for request
     *
     * http://domain.com/project/route
     *
     * when index.php ("script") is in project/index.php
     * path() returns just '/route', not '/project/route'.
     *
     * @return string
     */
    public function path()
    {
        $uri = urldecode(
            parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH)
        );

        if ($uri === '/')
            return '/';

        // Example:
        // SCRIPT_NAME = '/project/index.php'
        // REQUEST_URI = '/project/route'
        // dirname('/project/index.php') returns '/project'
        //
        $scriptName = $_SERVER['SCRIPT_NAME'];
        $lengthOfPrefix = strlen(dirname($scriptName));
        $path = substr($uri, $lengthOfPrefix);
        if ($path[0] !== '/')
            return '/' . $path;

        return $path;
    }


    /**
     * Does $_GET or $_POST have $key?
     *
     * @param array|string $key Can be an array, in which case this
     * method checks for existence of all keys in it.
     *
     * @return bool
     * @todo $_REQUEST ?
     */
    public function has($key)
    {
        if (! is_array($key))
            return isset($this->params[$key]);

        foreach ($key as $k) {
            if (isset($this->params[$k]))
                continue;

            return false;
        }

        return true;
    }


    /**
     * Retrieves a value from $_GET or $_POST (whichever has the key).
     *
     * Throws a RuntimeException if key is missing! Intended usage:
     *
     * <code>
     * if (! $request->has(['csrf_token', 'other'])) {
     *     // Handle invalid request.
     * }
     *
     * // At this point we are sure 'csrf_token' exists.
     * $token = $request->input('csrf_token');
     * </code>
     *
     * @param string $key           Lookup key.
     * @param string|array $default Default when key not found.
     *
     * @return string|array
     *
     * @throws RuntimeException if $key is missing and $default is not provided.
     *
     */
    public function input($key, $default = null)
    {
        if (isset($this->params[$key]))
            return $this->params[$key];

        if ($default !== null)
            return $default;

        throw new \RuntimeException('Missing request input: ' . $key);
    }


    /**
     * Returns all of the input (e.g. entire $_GET or $_POST).
     *
     */
    public function all()
    {
        return $this->params;
    }


    /**
     * Determines if a cookie is set on the request.
     *
     * @param string $key
     *
     * @return bool
     */
    public function hasCookie($key)
    {
        return isset($_COOKIE[$key]);
    }


    /**
     * Retrieves a cookie value. Throws RuntimeException if cookie
     * does not exist and $default is not provided.
     *
     * @param string $key
     *
     * @return string
     */
    public function cookie($key, $default = null)
    {
        if (isset($_COOKIE[$key]))
            return $_COOKIE[$key];

        if ($default !== null)
            return $default;

        throw new \RuntimeException('Missing request cookie: ' . $key);
    }


    /**
     * Verify the CSRF token value received through a custom header
     * is the same as the 'csrftoken' cookie value.
     *
     * This is the 'Double Submit Cookies' approach.
     *
     * @todo Hidden input field token!
     * @todo strict referer checking
     * @link https://docs.djangoproject.com/en/dev/ref/csrf/#how-it-works
     *
     */
    public function verifyCsrfToken()
    {
        // http://stackoverflow.com/questions/6287903/how-to-properly-add-csrf-token-using-php

        $hasHeader = $this->hasHeader('X-Csrf-Token');
        $hasCookie = $this->hasCookie('csrftoken');

        if (! $hasHeader || ! $hasCookie)
            throw new InvalidCsrfTokenException;

        $headerToken = $this->header('X-Csrf-Token');
        $cookieToken = $this->cookie('csrftoken');

        if (! hash_equals($cookieToken, $headerToken))
            throw new InvalidCsrfTokenException;
    }


    /**
     * Determines whether or not a header was received.
     *
     * @param string $key Case-insensitive.
     *
     * @return bool
     */
    public function hasHeader($key)
    {
        $key = strtolower($key);

        if (isset($this->headers[$key]))
            return true;

        return false;
    }


    /**
     * Retrieves a header from the request. Keys are case-insensitive.
     * Throws RuntimeException if header does not exist and $default is not provided.
     *
     * <code>
     * if ($request->hasHeader('user-agent'))
     *     $userAgent = $request->header('user-agent');
     * </code>
     *
     * @link http://stackoverflow.com/questions/5258977/are-http-headers-case-sensitive
     *
     * @param  string  $key
     * @param  mixed   $default
     *
     * @return string
     */
    public function header($key, $default = null)
    {
        $key = strtolower($key);

        if (isset($this->headers[$key]))
            return $this->headers[$key];

        if ($default !== null)
            return $default;

        throw new \RuntimeException('Missing header: ' . $key);
    }


    /**
     * Retrieve a file from the request.
     *
     * @param  string  $key
     * @param  mixed  $default
     * @return \SolveX\Http\UploadedFile
     */
    public function file($key, $default = null)
    {
        if (isset($_FILES[$key])) {
            $info = $_FILES[$key];
            return new UploadedFile($info['tmp_name'], $info['name'], $info['type'], $info['size'], $info['error']);
        }

        if ($default !== null)
            return $default;

        throw new \RuntimeException('Missing file: ' . $key);
    }
}
