
/**
 * @class Ext.ux.Calculator
 * @extends Ext.Component
 * Simple calculator class.
 * @constructor
 * Create a new Calculator
 * @param {Object} config The config object
 * @author Toby Stuart, Nullity, edits by Troy McCabe
 */
Ext.ux.Calculator = Ext.extend(Ext.Container, {
	/**
	 * @cfg {string} number 
	 * number placeholder for the value in the textbox
	 */
	number: '0',
	/**
	 * @cfg {string} num1 
	 * number placeholder for internal use by the calculator
	 */
	num1: '',
	
	/**
	 * @cfg {string} num2 
	 * number placeholder for internal use by the calculator
	 */
	num2: '',
	
	/**
	 * @cfg {string} operator 
	 * operator placeholder which operator is pressed
	 */
	operator: '',
	
	/**
	 * @cfg {string} memValue 
	 * number placeholder for the memory value in the textbox
	 */
	memValue: '0',
	
	/**
	 * @cfg {string} addToNum 
	 * [yes/no/reset] Whether to add the value if the user types another number if there is already in one in the textfield
	 */
	addToNum: 'no',
	
	/**
	 * @cfg {boolean} showOkButton 
	 * Whether to show the ok button at the bottom of the calculator
	 */
	showOkButton: false,
	
	/**
	 * @cfg {boolean} showTips 
	 * Whether to show the quicktips over the memory functions and such
	 */
	showTips: true,
	
	/**
	 * initialize the Component
	 */
	initComponent : function(){
		
		Ext.apply(this, {
			cls: 'ux-calc-container',
			//autoEl: {},
			layout: 'table',
			layoutConfig: {
				columns: 5,
				isValidParent : Ext.layout.FormLayout.prototype.isValidParent
			},
			defaultType: 'button',
			defaults: {
				scope: this
			},
			items: [ /*this.inputBox = new Ext.form.NumberField({
				colspan: 5,
				id: this.id,
				name: this.id,
				width: '100%',
				readOnly: true,
				cls: 'ux-calc-input',
				value: '0'
			}),*/ {
				text: '&#160;',
				cls: 'ux-calc-memstore',
				id: 'memStore_' + this.id,
				width: 30,
				disabled: true,
				handleMouseEvents: false
			}, {
				text: 'C',
				cls: 'ux-calc-memory',
				handler: this.clear,
				width: 30
			}, {
				text: 'CE',
				cls: 'ux-calc-memory',
				handler: this.clear,
				width: 30
			}, {
				text: 'BS',
				cls: 'ux-calc-memory',
				handler: this.clear,
				width: 30
			}, {
				text: '/',
				cls: 'ux-calc-operator',
				handler: this.operation,
				width: 30
			}, {
				text: 'MC',
				cls: 'ux-calc-memory',
				handler: this.memory,
				width: 30
			}, {
				text: '7',
				cls: 'ux-calc-digit',
				handler: this.enterDigit,
				width: 30
			}, {
				text: '8',
				cls: 'ux-calc-digit',
				handler: this.enterDigit,
				width: 30
			}, {
				text: '9',
				cls: 'ux-calc-digit',
				handler: this.enterDigit,
				width: 30
			}, {
				text: '*',
				cls: 'ux-calc-operator',
				handler: this.enterDigit,
				width: 30
			}, {
				text: 'MR',
				cls: 'ux-calc-memory',
				handler: this.memory,
				width: 30
			}, {
				text: '4',
				cls: 'ux-calc-digit',
				handler: this.enterDigit,
				width: 30
			}, {
				text: '5',
				cls: 'ux-calc-digit',
				handler: this.enterDigit,
				width: 30
			}, {
				text: '6',
				cls: 'ux-calc-digit',
				handler: this.enterDigit,
				width: 30
			}, {
				text: '-',
				cls: 'ux-calc-operator',
				handler: this.enterDigit,
				width: 30
			}, {
				text: 'MS',
				cls: 'ux-calc-memory',
				handler: this.memory,
				width: 30
			}, {
				text: '1',
				cls: 'ux-calc-digit',
				handler: this.enterDigit,
				width: 30
			}, {
				text: '2',
				cls: 'ux-calc-digit',
				handler: this.enterDigit,
				width: 30
			}, {
				text: '3',
				cls: 'ux-calc-digit',
				handler: this.enterDigit,
				width: 30
			}, {
				text: '+',
				cls: 'ux-calc-operator',
				handler: this.enterDigit,
				width: 30
			}, {
				text: 'M+',
				cls: 'ux-calc-memory',
				handler: this.memory,
				width: 30
			}, {
				text: '+/-',
				cls: 'ux-calc-misc',
				handler: this.plusminus,
				width: 30
			}, {
				text: '0',
				cls: 'ux-calc-digit',
				handler: this.enterDigit,
				width: 30
			}, {
				text: '.',
				cls: 'ux-calc-digit',
				handler: this.enterDot,
				width: 30
			}, {
				text: '=',
				cls: 'ux-calc-operator',
				handler: this.enterDigit,
				width: 30
			}]
		});

		//call the initComponent method of the superclass
		Ext.ux.Calculator.superclass.initComponent.call(this);
	},
	
	/**
	 * Function that fires on render of the component
	 * @param {object} container
	 * The container of the calculator
	 * @param {string} position
	 * The position of the component in the container
	 */
	onRender : function(container, position) {

		Ext.ux.Calculator.superclass.onRender.apply(this, arguments);

		//create the keymap
		this.keyMap = new Ext.KeyMap(this.el, {});

		//enable the keymap
		this.keyMap.enable();
		
	},

	/**
	 * get's the value from the calculator's textbox
	 * @returns value from textbox
	 */
	getValue : function() {
		return this.inputBox.getValue();
	},

	/**
	 * Set the textbox to the specified value
	 * @param {string} value
	 * The value to set the textbox to
	 */
	setValue : function(value) {
		this.number = value;
		this.inputBox.setValue(this.number);
	},

	/**
	 * Update the display to reflect actions
	 */
	updateDisplay : function() {
		//if it's infinity, reset it
		if (this.number == 'Infinity') {
			this.number = '0';
		}

		//set the value
		this.inputBox.setValue(this.number);
		this.inputBox.focus();
	},

	/**
	 * function when the user hits a digit
	 * @param {Button} The Button clicked
	 */
	enterDigit : function(b) {
		var n = b.text;

		//if it's set to add to the number, just add it
		if (this.addToNum == 'yes' || true) {
			this.number = this.inputBox.getValue();
			this.number += n;

			//handle decimal places
			if (this.number.charAt(0) == 0 && this.number.indexOf('.') == -1) {
				this.number = this.number.substring(1);
			}
		}
		//not set to add the number
		else {
			//if it's set to reset, clear first
			if (this.addToNum == 'reset') {
				this.reset();
			}

			//reset the number
			this.number = n;
			this.addToNum = 'yes';
		}
		
		//update the display
		this.updateDisplay();
	},

	/**
	 * function when the user hits a period
	 */
	enterDot : function() {
		//if set to just add to the number, append the dot
		if (this.addToNum == 'yes') {
			//if it's not -1
			if (this.number.indexOf('.') != -1) {
				return;
			}

			this.number += '.';
		}
		//else reset
		else {
			//reset the field
			if (this.addToNum == 'reset') {
				this.reset();
			}

			//set the number
			this.number = '0.';
			this.addToNum = 'yes';
		}

		//update the display
		this.updateDisplay();
	},

	/**
	 * function when plus or minus is pressed
	 */
	plusminus : function() {
		//if there's nothing in the box, skip
		if (this.number == '0') {
			return;
		}

		//else, update the number (do the operation)
		this.number = (this.number.charAt(0) == '-') ? this.number.substring(1) : '-' + this.number;
		
		//update the display
		this.updateDisplay();
	},

	/**
	 * Function to reset all the placeholders
	 */
	reset : function() {
		//clear all back to orig
		this.number = '0';
		this.addToNum = 'no';
		this.num1 = '';
		this.num2 = '';
		this.operator = '';
	},

	/**
	 * Function to fire on the clear buttons
	 * @param {string} o
	 * the button that was pressed
	 */
	clear : function(b) {
		//call function depending on button
		switch(b.text) {
			case 'C':
				this.clearAll();
				break;
			case 'CE':
				this.clearEntry();
				break;
			case 'BS':
				this.backspace();
				break;
			default:
				break;
		}
	},

	/**
	 * Function to clear all from the stuff
	 */
	clearAll : function() {
		//reset it
		this.reset();
		
		//update the display
		this.updateDisplay();
	},

	/**
	 * Function to clear just the entry
	 */
	clearEntry : function() {
		//set the number to 0
		this.number = '0';
		
		//don't add to the number
		this.addToNum = 'no';
		
		//update the display
		this.updateDisplay();
	},

	/**
	 * function to remove just one digit
	 */
	backspace : function() {
		//convert to string
		var n = this.number + '';

		//if it's 0, skip
		if (n == '0') {
			return;
		}

		//remove one char
		this.number = n.substring(0, n.length-1);
		
		//update the display
		this.updateDisplay();
	},

	/**
	 * Function to call correct memory function depending on button
	 * @param {string} 0
	 * the button that was pressed
	 */
	memory : function(b) {
		//switch & call correct function depending on button
		switch(b.text) {
			case 'M+':
				this.memStore(true);
				break;
			case 'MS':
				this.memStore();
				break;
			case 'MR':
				this.memRecall();
				break;
			case 'MC':
				this.memClear();
				break;
			default:
				break;
		}
	},

	/**
	 * Function to store the value in memory
	 * @param {boolean} add
	 * whether to add it to the current stored value
	 */
	memStore : function(add) {
		//if the number is 0, or not there, skip
		if (!this.number || this.number == '0') {
			return;
		}
		//otherwise
		else {
			//set the value (add if add is true)
			this.memValue = (add === true) ? this.calculate(this.number, this.memValue, '+') : this.number;

			//set the memButton to show there is something in memory
			var memButton = Ext.getCmp('memStore_' + this.id);
			memButton.setText('M');
			memButton.toggle(true);

			//register the quicktip if they are set to be shown
			if (this.showTips) {
				Ext.QuickTips.register({
					target: memButton.container.dom,
					text: 'Memory: <b>' + this.memValue + '</b>'
				});
			}
		}
	},

	/**
	 * Function to recall the value from memory
	 */
	memRecall : function() {
		//if there is a value in memory
		if (this.memValue != '0') {
			//set the number
			this.number = this.memValue;

			//set the other numbers too
			if (this.num1) {
				this.num2 = this.memValue;
			}

			//update the display
			this.updateDisplay();
		}
	},

	/**
	 * Function to clear the value from memory
	 */
	memClear : function() {
		//set the memory value to 0
		this.memValue = '0';
		
		//clear the memory div
		var memButton = Ext.getCmp('memStore_' + this.id);
		memButton.setText('&#160;');
		memButton.toggle(false);

		//remove the quicktip
		if (this.showTips) {
			Ext.QuickTips.unregister(memButton.container.dom);
		}
	},

	/**
	 * function to check the accuracy of the result
	 * @param {double} result
	 * the result from the operation
	 * @returns the result of the mathematical operation
	 */
	accuracyCheck : function(result) {
		//set up variables to be used
		var i, n, j, k;
		var check;

		//loop through 9
		for (i = 0; i < 9; i++) {
			//set variables to be used to check the accuracy
			check = result * Math.pow(10, i);
			k = i + 1;
			n = Math.abs(Math.round(check) - check);
			j = Math.pow(10, -(12-i));

			//if n < j, then return the correct number
			if (n < j) {
				return Math.round(check) * Math.pow(10, -i);
			}
		}

		//if n > j, then return the result (it was correct)
		return result;
	},

	/**
	 * Function to calculate the value
	 * @param {string} o1
	 * old value
	 * @param {string} o2
	 * new value
	 * @param {string} op
	 * operation to perform
	 * @returns the result of the operation
	 */
	calculate : function(o1, o2, op) {
		//create the result variable
		var result;

		//if you want to =
		if (op == '=') {
			//set the result
			result = o1 = o2;
			o2 = '';
		}
		//+, -, etc.
		else {
			//eval the value
			result = eval('o1 + op + o2');
			result = eval(result);
		}

		//return the result
		return result;
	},

	/**
	 * Function to perform the operation on the values
	 * @param {Button} The Buton pressed.
	 * The operation to perform
	 */
	operation : function(b) {
		var op = b.text;

		//if num1 is blank
		if (this.num1 == '' && typeof(this.num1) == 'string') {
			//set num1 = the current textbox
			this.num1 = parseFloat(this.number);
			
			//set the operator
			this.operator = op;
			
			//change the option
			this.addToNum = 'no';
		}
		//it has something
		else {
			//if set to add to the number
			if (this.addToNum == 'yes') {
				//set num2 to the current value
				this.num2 = parseFloat(this.number);
				
				//call the calculate function
				this.num1 = this.calculate(this.num1, this.num2, this.operator);
				
				//set the number to the accuracy check's value
				this.number = this.accuracyCheck(this.num1) + '';
				
				//change the display
				this.updateDisplay();
				
				//set the operator
				this.operator = op;
				
				//set the option
				this.addToNum = 'no';
			}
			//it's not set to add to the number
			else {
				//change the options
				this.operator = op;
				this.addToNum = 'no';
			}
		}
	},

	/**
	 * Function to equal the two values together
	 */
	equals : function() {
		//if set to add to number
		if (this.addToNum == 'yes') {
			//if num1 is blank
			if (this.num1 == '' && typeof(this.num1) == 'string') {
				//change the options (set values)
				this.operator = '=';
				this.num1 = parseFloat(this.number);
				this.addToNum = 'no';
			}
			//num1 has a value
			else {
				//do the math
				this.num2 = parseFloat(this.number);
				this.num1 = this.calculate(this.num1, this.num2, this.operator);
				this.number = this.accuracyCheck(this.num1) + '';
				
				//change the options (set values)
				this.updateDisplay();
				this.addToNum = 'reset';
			}
		}
		//not set to add to number
		else {
			//num1 is null - skip
			if (this.num1 == '' && typeof(this.num1) == 'string') {
				return;
			}
			//num1 has a value
			else {
				//if num2 has no value, set it to num1's value
				if (this.num2 == '' && typeof(this.num2) == 'string') {
					this.num2 = this.num1;
				}

				//do the math on the numbers
				this.num1 = this.calculate(this.num1, this.num2, this.operator);
				this.number = this.accuracyCheck(this.num1) + '';
				
				//change the config options
				this.updateDisplay();
				this.addToNum = 'reset';
			}
		}
	},
	
	/**
	 * Function to align to the specified position in the element
	 */
	alignTo : function() {
		this.el.alignTo.apply(this.el, arguments);
	},

	/**
	 * Function to handle the click of the ok button
	 */
	ok : function() {
		//fire the 'hide' event
		this.fireEvent('hide', this);
	}
});

//Register the xtype
Ext.reg('calculator', Ext.ux.Calculator);

/**
 * @class Ext.ux.CalculatorItem
 * @constructor
 * @extends Ext.menu.Adapter
 * @author Troy McCabe
 * Creates the calculator menu item to be used in a menu
 */
 /*
Ext.ux.CalculatorItem = function(config){
	//call the parent's constructor
	Ext.ux.CalculatorItem.superclass.constructor.call(this, new Ext.ux.Calculator(config), config);
	
	//set the calculator for reference
	this.calculator = this.component;
};

//do the actual extension
Ext.extend(Ext.ux.CalculatorItem, Ext.menu.Adapter);
*/
/**
 * @class Ext.ux.CalculatorMenu
 * @constructor
 * @extends Ext.menu.Menu
 * @author Troy McCabe
 * Creates the calculator menu
 */
/*Ext.ux.CalculatorMenu = function(config){
	//call the parent constructor
	Ext.ux.CalculatorMenu.superclass.constructor.call(this, config);
	
	//set the menu to plain
	this.plain = true;
	
	//set ci to the calculator item
	config.forceLayout = true;
	this.calculator = new Ext.ux.Calculator(config);
	this.calculator.doLayout();
	//add it
	this.add(this.calculator);
	
	//set the calculator in this object for reference by the menu
	//this.calculator = ci.calculator;
};*/

//do the actual extension
Ext.ux.CalculatorMenu = Ext.extend(Ext.menu.Menu, {
	//set the class to a x-calculator-menu
	cls:'x-calculator-menu',
	layout:'auto',
	width : 167,
	
	initComponent: function(){
      
        Ext.apply(this, {
            plain: true,
            showSeparator: false,
            items: [this.calculator = new Ext.ux.Calculator(Ext.apply({
                forceLayout:true
            }, this.initialConfig)) ]
        });
		
		
        Ext.ux.CalculatorMenu.superclass.initComponent.call(this);
		this.calculator.initComponent();
       // this.relayEvents(this.picker, ["select"]);
    },
	
	//override the before destroy method to destroy the calculator component that was created for this menu
	beforeDestroy : function() {
		this.calculator.destroy();
	}
});

Ext.ux.CalcField = Ext.extend(Ext.form.TriggerField,{
	
	 defaultAutoCreate : {tag: "input", type: "text", size: "10", autocomplete: "off"},
	 onTriggerClick : function(){
        if(this.disabled){
            return;
        }
        
        if(this.menu == null){
            this.menu = new Ext.ux.CalculatorMenu({
                hideOnClick: false,
                focusOnSelect: false,
                inputBox : this
            });
            Ext.menu.MenuMgr.unregister(this.menu);
        }
        if(this.menu.isVisible())
        {
        	this.menu.hide();	
        }
        else
        {
        	this.menu.show(this.el, "tl-br?");
        }
       
    }
	
});
Ext.reg('calcfield',Ext.ux.CalcField);

