<?php

namespace App\Tests;

use PHPUnit\Framework\TestCase;
use App\Services\QuestionService;
use Illuminate\Support\Collection;
use App\Exceptions\RuntimeException;
use App\Entities\Question;
use \Mockery as m;

class QuestionServiceTest extends TestCase
{
    public function setupMocks()
    {
        $this->questionRepository    = m::mock('App\Repositories\QuestionRepository');
        $this->projectRepository     = m::mock('App\Repositories\ProjectRepository');
        $this->warpitConfigService   = m::mock('App\Services\WarpitConfigService');
        $this->structureTableService = m::mock('App\Services\StructureTableService');
        $this->answersConvertService = m::mock('App\Services\AnswersConvertService');

        $this->testAnswerString = '1 Yes/2 No';
        $this->testQuestion = Question::createQuestion((object)[
          'tip_sql'       => 1,
          'id'            => 1,
          'questionText'  => "Do you like weather today?",
          'answersString' => $this->testAnswerString,
        ]);
    }

    public function test_get()
    {
        $this->setupMocks();

        $this
          ->questionRepository
          ->shouldReceive('get')
          ->once()
          ->andReturn(
              collect([(object)[
          'id'               => '12',
          'tip_sql'         => '1',
          'q_name'            => 'q1',
          'variableTypeBase' => '1',
          'variableTypeOpen' => '4',
          'answersString'    => '1 Yes/2 No',
        ]]),
        [],
        collect([(object)[
          'id'                => '12',
          'tip_sql'           => '1',
          'q_name'      => 'q1',
          'variableTypeBase'  => '1',
          'variableTypeOpen'  => '4',
          'answersString'     => '1 Yes/2 No',
      ], (object)[
          'id'               => '12',
          'tip_sql'          => '1',
          'q_name'           => 'q1',
          'variableTypeBase' => '1',
          'variableTypeOpen' => '4',
          'answersString'    => '1 Yes/2 No',]])
          );

        $questionService = new QuestionService(
            $this->questionRepository,
                                                $this->projectRepository,
                                                $this->warpitConfigService,
                                                $this->structureTableService,
                                                $this->answersConvertService
        );

        $question = $questionService->get('question', 'project', 12);

        $this->assertInstanceOf('App\Entities\SingleResponse', $question);
        $this->assertSame('12',    $question->id);
        $this->assertSame('q1', $question->questionName);
        $this->assertSame('1',  $question->type);

        $question = $questionService->get('question', 'project', 12);

        $this->assertNull($question);

        $question = $questionService->get('question', 'project', 12);
        $this->assertInternalType('array',$question);
        $this->assertCount(2, $question);

    }

    public function test_getIfExists()
    {
        $questionService = m::mock('App\Services\QuestionService')->makePartial();

        $questionService->shouldReceive('get')->andReturn(null);

        $this->expectException(RuntimeException::class);
        $question = $questionService->getIfExists('question', 'project', 42);

    }

    public function test_questionExists()
    {
        $this->setupMocks();

        $questionService = m::mock('App\Services\QuestionService')->makePartial();

        $questionService->shouldReceive('getIfExists')->twice()->andReturn(null, $this->testQuestion);

        $response = $questionService->questionExists('question', 'project', 42);
        $this->assertFalse($response);

        $response = $questionService->questionExists('question', 'project', 42);
        $this->assertTrue($response);
    }

    public function test_getPropertyValue()
    {
        $this->setupMocks();

        $questionService = m::mock('App\Services\QuestionService')->makePartial();

        $questionService->shouldReceive('getIfExists')->andReturn($this->testQuestion);

        $id = $questionService->getPropertyValue('question', 'project', 'id', 42);
        $this->assertSame(1, $id);

        $this->expectException(RuntimeException::class);
        $zid = $questionService->getPropertyValue('question', 'project', 'zid', 42);

    }

    public function test_getQuestionText()
    {
        $this->setupMocks();

        $questionService = m::mock('App\Services\QuestionService')->makePartial();

        $questionService->shouldReceive('getPropertyValue')->andReturn('Do you like weather today?');

        $questionText = $questionService->getQuestionText('question', 'projectName', 42);

        $this->assertSame('Do you like weather today?', $questionText);
    }

    public function test_addQuestionText()
    {
        $this->setupMocks();

        $this
          ->questionRepository
          ->shouldReceive('getFromLanguageTable')
          ->twice()
          ->andReturn('oldtext', null);

        $this
          ->questionRepository
          ->shouldReceive('updateLanguageTable')
          ->once()
          ->andReturn('update');

        $this
          ->questionRepository
          ->shouldReceive('insertIntoLanguageTable')
          ->once()
          ->andReturn('insert');

        $questionService = m::mock('App\Services\QuestionService',
                                [
                                  $this->questionRepository,
                                  $this->projectRepository,
                                  $this->warpitConfigService,
                                  $this->structureTableService,
                                  $this->answersConvertService
                                ])->makePartial();

        $questionService->shouldReceive('getIfExists')->twice()->andReturn($this->testQuestion);

        $response = $questionService->addQuestionText('question', 'project', 'text', 42);
        $this->assertSame('update', $response);

        $response = $questionService->addQuestionText('question', 'project', 'text', 42);
        $this->assertSame('insert', $response);

    }
}
