<?php

namespace App\Tests;

use PHPUnit\Framework\TestCase;
use App\Services\AnswerService;
use App\Entities\Question;
use App\Exceptions\RuntimeException;
use \Mockery as m;

class AnswerServiceTest extends TestCase
{
    public function setupMocks()
    {
        $this->questionRepository    = m::mock('App\Repositories\QuestionRepository');
        $this->projectRepository     = m::mock('App\Repositories\ProjectRepository');
        $this->warpitConfigService   = m::mock('App\Services\WarpitConfigService');
        $this->structureTableService = m::mock('App\Services\StructureTableService');
        $this->answersConvertService = m::mock('App\Services\AnswersConvertService');
        $this->questionService       = m::mock('App\Services\QuestionService');

        $this->testAnswerString = '1 Yes/2 No';
        $this->testAnswersArray = [
          (object)[
                    'id' => 1,
                    'label' => 'Yes'
                  ],
          (object)[
                    'id' => 2,
                    'label' => 'No'
                    ]
          ];
        $this->testQuestion = Question::createQuestion((object)[
          'tip_sql'       => 1,
          'id'            => 1,
          'answersString' => $this->testAnswerString,
        ]);
    }

    public function test_getAnswersString()
    {
        $this->setupMocks();

        $this
          ->questionService
          ->shouldReceive('getIfExists')
          ->once()
          ->andReturn($this->testQuestion);

        $this
          ->questionRepository
          ->shouldReceive('getFromLanguageTable')
          ->once()
          ->andReturn($this->testAnswerString);

        $answersService = new AnswerService(
            $this->questionRepository,
                                            $this->projectRepository,
                                            $this->warpitConfigService,
                                            $this->structureTableService,
                                            $this->answersConvertService,
                                            $this->questionService
        );

        $answersString = $answersService->getAnswersString('questionname', 'projectName', 12);
        $this->assertInternalType('string', $answersString);
        $this->assertSame('1 Yes/2 No', $answersString);
    }

    public function test_getAnswers()
    {
        $this->setupMocks();

        $this
          ->questionService
          ->shouldReceive('getIfExists')
          ->once()
          ->andReturn($this->testQuestion);

        $this
          ->questionRepository
          ->shouldReceive('getFromLanguageTable')
          ->once()
          ->andReturn($this->testAnswerString);

        $this
          ->answersConvertService
          ->shouldReceive('toArray')
          ->once()
          ->andReturn($this->testAnswersArray);

        $answersService = new AnswerService(
            $this->questionRepository,
            $this->projectRepository,
            $this->warpitConfigService,
            $this->structureTableService,
            $this->answersConvertService,
            $this->questionService
        );

        $answersArray = $answersService->getAnswers('questionname', 'projectName', 12);
        $this->assertInternalType('array', $answersArray);
    }

    public function test_addAnswer()
    {
        $this->setupMocks();

        $this
          ->questionService
          ->shouldReceive('getIfExists')
          ->andReturn($this->testQuestion);

        $this
          ->answersConvertService
          ->shouldReceive('toArray')
          ->once()
          ->andReturn($this->testAnswersArray);

        $this
          ->answersConvertService
          ->shouldReceive('toString')
          ->andReturn($this->testAnswerString);

        $this
          ->questionRepository
          ->shouldReceive('getFromLanguageTable')
          ->once()
          ->andReturn([]);

        $this
          ->questionRepository
          ->shouldReceive('updateLanguageTable')
          ->once();

        $answersService = new AnswerService(
              $this->questionRepository,
              $this->projectRepository,
              $this->warpitConfigService,
              $this->structureTableService,
              $this->answersConvertService,
              $this->questionService
          );

        $newArray = $answersService->addAnswer('q1', 'project', 3, 'Maybe', 42);

        $this->assertCount(3, $newArray);
        $this->assertSame(3, $newArray[2]->id);
        $this->assertSame('Maybe', $newArray[2]->label);
    }

    public function test_addAnswersWrongParams()
    {
        $this->setupMocks();
       $this->expectException(RuntimeException::class);

        $this
          ->questionService
          ->shouldReceive('getIfExists')
          ->andReturn($this->testQuestion);

        $this
          ->answersConvertService
          ->shouldReceive('toArray')
          ->once()
          ->andReturn($this->testAnswersArray);

        $answersService = new AnswerService(
          $this->questionRepository,
          $this->projectRepository,
          $this->warpitConfigService,
          $this->structureTableService,
          $this->answersConvertService,
          $this->questionService
      );

        $newArray = $answersService->addAnswer('q1', 'project', 1, 'Maybe', 42);
    }

    public function test_deleteAnswers()
    {
        $this->setupMocks();

        $this
          ->questionService
          ->shouldReceive('getIfExists')
          ->andReturn($this->testQuestion, $this->testQuestion);

        $this
          ->answersConvertService
          ->shouldReceive('toArray')
          ->andReturn($this->testAnswersArray);

        $this
          ->answersConvertService
          ->shouldReceive('toString')
          ->andReturn($this->testAnswerString);

        $this
          ->questionRepository
          ->shouldReceive('getFromLanguageTable')
          ->once()
          ->andReturn([]);

        $this
          ->questionRepository
          ->shouldReceive('updateLanguageTable')
          ->once();

        $answersService = new AnswerService(
            $this->questionRepository,
            $this->projectRepository,
            $this->warpitConfigService,
            $this->structureTableService,
            $this->answersConvertService,
            $this->questionService
        );

        $newArray = $answersService->deleteAnswer('question', 'project', 1, 42);
        $this->assertCount(1, $newArray);
        $this->assertSame(2, $newArray[1]->id);

        $this->expectException(RuntimeException::class);
        $response = $answersService->deleteAnswer('question', 'project', 6, 42);

    }

    public function test_addAnswerString()
    {
        $this->setupMocks();

        $this
          ->questionRepository
          ->shouldReceive('getFromLanguageTable')
          ->andReturn([], null);

        $this
          ->questionRepository
          ->shouldReceive('updateLanguageTable')
          ->once()
          ->andReturn('update');

        $this
          ->questionRepository
          ->shouldReceive('insertIntoLanguageTable')
          ->once()
          ->andReturn('insert');

        $this
          ->questionRepository
          ->shouldReceive('deleteFromLanguageTable')
          ->once()
          ->andReturn('delete');

        $answersService = new AnswerService(
            $this->questionRepository,
            $this->projectRepository,
            $this->warpitConfigService,
            $this->structureTableService,
            $this->answersConvertService,
            $this->questionService
        );


        $response = $answersService->addAnswersString(12, 'fake', null, 42);
        $this->assertSame('delete', $response);

        $response = $answersService->addAnswersString(12, 'fake', $this->testAnswerString, 42);
        $this->assertSame('update', $response);

        $response = $answersService->addAnswersString(12, 'fake', $this->testAnswerString, 42);
        $this->assertSame('insert', $response);

    }
}
