<?php

namespace App\Services;

use App\Repositories\ProjectRepository;
use App\Repositories\WarpitConfigRepository;
use App\Repositories\SystemDefaultQuestionsRepository;
use App\Transformers\ProjectsTransformer;
use Carbon\Carbon;
use App\Entities\LanguageTextType;
use App\Entities\Project;

class ProjectService
{
    protected $projectsTransformer = null;

    public function __construct(ProjectRepository $projectRepository,
                                WarpitConfigService $config,
                                UniqueLinksService $uniqueLinksService,
                                SystemDefaultQuestionsRepository $systemDefaultQuestionsRepository,
                                StructureTableService $structureTableService,
                                ProjectsTransformer $projectsTransformer)
    {
        $this->projectRepository = $projectRepository;
        $this->config = $config;
        $this->uniqueLinksService = $uniqueLinksService;
        $this->systemDefaultQuestionsRepository = $systemDefaultQuestionsRepository;
        $this->structureTableService = $structureTableService;

        $this->projectsTransformer = $projectsTransformer;
    }

    public function create($name, $languageId, $subFolder = 0, $createUser = 'warpit-api')
    {
        if ($this->projectExists($name)) {
            throw new \RuntimeException('Project already exists!');
        }

        $orderBy = $this->projectRepository->getMaxOrder() + 1;
        $projectId = $this
            ->projectRepository
            ->insertIntoProjectTreeTable($name, $orderBy, $subFolder);

        $endDateInterval = $this->config->getConfig('webcati', 'end_data_default_interval', 'P1Y');
        $active = $this->config->getConfig('webcati', 'premission_for_interviewing', '1');
        $lockAppToUser = $this->config->getConfig('webcati', 'default_lock_app_to_user', '1');

        $dt = Carbon::now();
        $endDate = $dt->add(new \DateInterval($endDateInterval))->format('Y-m-d');
        $keyProj = substr(uniqid($projectId . '-'), 0, -2);

        $project = [
            'id' => $projectId,
            'proj_name' => $name,
            'key_proj' => $keyProj,
            'old' => 'N',
            'creat_name' => $createUser,
            'created' => $dt->format("d.m.Y - H:i"),
            'modi_name' => $createUser,
            'modify' => $dt->format("d.m.Y - H:i"),
            'rotation' => 'N',
            'set_use_autodial' => false,
            'language' => $languageId,
            'start_date' => $dt->format('Y-m-d'),
            'end_date' => $endDate,
            'active' => $active,
            'lock_app_to_user' => $lockAppToUser,
        ];

        $this->projectRepository->insertIntoProjectTable($project);
        $this->projectRepository->insertLanguage($languageId, $projectId);

        $this->projectRepository->createProjectTable($name);
        $this->projectRepository->createTreeTable($name);
        $this->projectRepository->createScriptTable($name);
        $this->projectRepository->createLanguageTable($name);

        $this->uniqueLinksService->createPopup($projectId);

        $this->fillSystemDefaultQuestions($name, $languageId);

        $this->structureTableService->create($name);

        return $this->get($name);
    }

    public function projectExists($id)
    {
        if (ctype_digit($id)) {
            return $this->projectRepository->getProjectName($id) !== null;
        }

        if (is_string($id)) {
            return $this->projectRepository->getProjectId($id) !== null;
        }
    }

    public function get($name)
    {
        $projectData = $this->projectRepository->getProject($name);

        $project = new Project;
        $project->id = $projectData->id;
        $project->projectName = $projectData->proj_name;
        $project->longProjectName = $projectData->naziv_proj;

        return $project;
    }

    public function getProjectById($projectId)
    {
        $projectData = $this->projectRepository->get($projectId);

        return $this->projectsTransformer->transform(collect([$projectData]));
    }

    public function delete($projectName)
    {
        if (!$this->projectExists($projectName)) {
            return 'Project doesn\'t exists';
        }

        $projectId = $this->projectRepository->getProjectId($projectName);

        $this->projectRepository->dropProjectTable($projectName);
        $this->projectRepository->dropProjectTreeTable($projectName);
        $this->projectRepository->dropProjectScriptTable($projectName);
        $this->projectRepository->dropProjectLanguageTable($projectName);
        $this->projectRepository->deleteFromProjectTable($projectId);
        $this->projectRepository->deleteFromProjectsTreeTable($projectId);
        $this->projectRepository->deleteLanguage($projectId);
        $this->uniqueLinksService->deletePopup($projectId);
        $this->structureTableService->delete($projectName);
    }

    public function createFolder($name)
    {
    }

    public function getUnactiveProjects()
    {
        return $this->projectRepository->getUnactiveProjects();
    }

    public function getStructureTable($projectId)
    {
        return $this->projectRepository->getStructureTable($projectId);
    }

    private function fillSystemDefaultQuestions($name, $languageId)
    {
        $defaultQuestions = $this->systemDefaultQuestionsRepository->get();
        $disableConfig = $this->config->getConfig('webcati', 'default_hide_start_end', '0');

        list($disable['_Start'], $disable['_End']) = explode(',', $disableConfig);

        foreach ($defaultQuestions as $defaultQuestion) {
            $qName = $defaultQuestion->q_name;

            $disabled = key_exists($qName, $disable) ? $disable[$qName] : 0;

            $question = [
                'q_name' => $qName,
                'q_pos' => $defaultQuestion->q_pos,
                'tip_sql' => $defaultQuestion->tip_sql,
                'goto_sql' => $defaultQuestion->goto_sql,
                'disable_question' => $disabled,
            ];

            $questionId = $this->projectRepository->insertSystemDefaultQuestions($name, $question);
            $text = [
                'id_language' => $languageId,
                'id_question' => $questionId,
                'id_text_type' => LanguageTextType::QUESTION_TEXT,
                'lngText' => $defaultQuestion->default_text,
            ];

            $this->projectRepository->insertSystemDefaultLanguage($name, $text);
        }
    }


}
