<?php

namespace App\Services;

use App\Repositories\PanelistRepository;
use App\Exceptions\RuntimeException;
use App\Repositories\PanelistStatusRepository;
use App\Transformers\PointsTransformer;

class PanelistService
{

    protected $panelistRepository;

    protected $panelistStatusService;

    protected $panelistFieldsService;

    protected $pointsTransformer;

    protected $config;

    public function __construct(PanelistRepository $panelistRepository,
                                WarpitConfigService $config,
                                PanelistStatusService $panelistStatusService,
                                PanelistFieldsService $panelistFieldsService,
                                PointsTransformer $pointsTransformer
    )
    {
        $this->panelistStatusService = $panelistStatusService;
        $this->panelistFieldsService = $panelistFieldsService;
        $this->panelistRepository    = $panelistRepository;
        $this->pointsTransformer     = $pointsTransformer;

        $this->config = $config;
    }

    public function get($panelistId)
    {
        return $this->panelistRepository->get($panelistId);
    }

    public function update($panelistId, $fields)
    {
        if (!$this->exists($panelistId)) {
            throw new RuntimeException("Panelist [$panelistId] doesn't exists!", 400);
        }
        if (!$this->panelistFieldsService->exits(array_keys($fields))) {
            throw new RuntimeException("One of the fields doesn't exists!", 400);
        }

        $this->panelistRepository->update($panelistId, $fields);
    }

    public function allowedToLogin($panelistId)
    {
        $notAllowStatuses = ['9', '8', '7', '6'];

        $status = $this->panelistStatusService->get($panelistId)->first();

        if ($status === null) {
            //we don't have defined status so this means we can login
            return true;
        }

        if (in_array($status->definedStatus, $notAllowStatuses)) {
            return false;
        }

        return true;
    }

    public function hasStatus($panelistId)
    {
        $statuses = $this->panelistStatusService->get($panelistId);
        return $statuses->first() !== null;

    }

    public function exists($panelistId)
    {
        $panelist = $this->panelistRepository->get($panelistId);

        return $panelist !== null;
    }

    public function addStatus($panelistId, $status)
    {
        if (!$this->exists($panelistId)) {
            throw new RuntimeException("Panelist [$panelistId] doesn't exists!", 400);
        }

        if ($this->hasStatus($panelistId, $status)) {
            return $this->panelistStatusService->update($panelistId, $status);

        }

        return $this->panelistStatusService->insert($panelistId, $status);

    }

    public function addRemovalRequestStatus($panelistId)
    {

        $personalData = (bool)$this->config->getConfig('removal-request', 'remove-personal-data', 0);
        $fromGroups   = (bool)$this->config->getConfig('removal-request', 'remove-from-groups', 0);
        $deletedGroup = (int)$this->config->getConfig('removal-request', 'removed-panelists-group', 0);

        $this->addStatus($panelistId, 6);

        if ($personalData === true) {
            $this->removePersonalData($panelistId);
        }

        if ($fromGroups === true) {
            $this->removeFromGroups($panelistId);
        }

        if ($deletedGroup > 0) {
            $this->addToGroup($panelistId, $deletedGroup);
        }
    }

    public function destroyEmail($panelistId)
    {
        $panelist = $this->panelistRepository->get($panelistId);
        $email = $panelist->email;

        $newEmail = str_replace('@', '[deleted]', $email);

        $this->panelistRepository->update($panelistId, ['email' => $newEmail]);
    }

    public function removePersonalData($panelistId)
    {
        $destroyEmail = (bool)$this->config->getConfig('removal-request', 'destroy-email', 0);

        $fields = $this->panelistRepository->getFields();

        $updateData = [];
        foreach ($fields as $field) {
            $updateData[$field->field_name] = null;
        }

        $this->panelistRepository->update($panelistId, $updateData);

        if ($destroyEmail === true) {
            $this->destroyEmail($panelistId);
        }
    }

    public function addToGroup($panelistId, $groupId)
    {
        $this->panelistRepository->addToGroup($panelistId, $groupId);
    }

    public function removeFromGroups($panelistId)
    {
        $this->panelistRepository->removeFromGroups($panelistId);
    }

    public function getFromGroup($groupId)
    {
        return $this->panelistRepository->getFromGroup($groupId);
    }

    public function getPoints($panelistId)
    {
        $points = $this->panelistRepository->getPoints($panelistId);

        return $this->pointsTransformer->transform($points);
    }

    public function getFieldWithLabels($field_name)
    {
        $field = $this->panelistRepository->getFieldByName($field_name);
        $field->labels = $this->panelistRepository->getFieldLabels($field->id);
        return $field;
    }
}
