<?php

namespace App\Controllers\V1;

use SolveX\Interfaces\RequestInterface;
use RuntimeException;
use DB;
use Response;

class UserController
{
    protected $request;

    public function __construct(RequestInterface $request)
    {
        $this->request = $request;
    }

    /**
     * This is used in registration.
     */
    public function addUser()
    {
        $email    = $this->request->input('email');
        $password = $this->request->input('password');

        if ($this->emailInUse($email))
            Response::abort(400, 'Email already in use!');

        $userId = $this->insertUserGetId($email, $password);
        return ['id' => $userId, 'email' => $email];
    }

    /**
     * User email must be unique.
     */
    private function emailInUse($email)
    {
        $userTable = env('DB_USER_TABLE', '_Interviewer');
        $panelDatabase  = env('TABLE_WARPIT_USERSPOOL');

        $userId = DB::selectOne("SELECT id FROM $panelDatabase . $userTable WHERE email = ?", [$email]);

        return $userId !== null;
    }

    /**
     * Insert a new user into user database table. User is uniquely identified
     * by email.
     */
    private function insertUserGetId($email, $password)
    {
        $userTable = env('DB_USER_TABLE', '_Interviewer');
        $panelDatabase  = env('TABLE_WARPIT_USERSPOOL');

        $hash = password_hash($password, PASSWORD_DEFAULT, ['cost' => 10]);

        DB::insert(
            "INSERT INTO $panelDatabase . $userTable (email, active, created, password) VALUES(?, 1, NOW(), ?)",
            [$email, $hash]
        );

        $userId = DB::selectOne('SELECT LAST_INSERT_ID() as id');
        return $userId->id;
    }

    /**
     * Retrieve user info.
     */
    public function getUser($userId)
    {
        $user = $this->getUserPoints($userId);
        $user['id'] = $userId;
        return $user;
    }

    /**
     * e.g. GET /users/search?id=42
     *
     * {
     *    "user": {
     *      "id": "42",
     *      "email": "..............",
     *      "active": "1",
     *      "created": "2015-12-01 09:07:52"
     *    }
     * }
     *
     */
    public function searchUsers()
    {
        if ($this->request->has('id'))
            return $this->findUserBy('id', $this->request->input('id'));
        else if ($this->request->has('email'))
            return $this->findUserBy('email', $this->request->input('email'));

        Response::abort(400, 'Invalid search parameters!');
    }

    private function findUserBy($column, $value)
    {
        $userTable = env('DB_USER_TABLE', '_Interviewer');
        $panelDatabase  = env('TABLE_WARPIT_USERSPOOL');

        $user = DB::selectOne("SELECT id, email, active, created FROM $panelDatabase . $userTable WHERE $column = ?", [$value]);

        if ($user === null)
            Response::abort(400, 'No results!');

        return ['user' => $user];
    }

    /**
     * Retrieves surveys accessible by user
     *
     * GET method for REST service - get all surveys that user can edit.
     *
     * @todo
     */
    public function getUserSurveys($userId)
    {
      $webcatiBaseTable = env('TABLE_WARPIT_WEBCATI_BASE');

      //Sql for surveys without points of survey
      /*$mySurveys = DB::select("SELECT loginSID, id_project, proj_name as title, id_user, id_site, start_date, end_date as expires
                               FROM $webcatiBaseTable._www_ids ids
                               LEFT JOIN $webcatiBaseTable._Projects p
                               ON ids.id_project = p.id
                               WHERE user_type = 2 AND id_user = ?", [$userId]);*/


      $mySurveys = DB::select("SELECT loginSID, ids.id_project, proj_name as title, id_user, id_site, start_date, end_date as expires, MIN(setPoints) as minPoints, SUM(setPoints) as maxPoints, p.onboarding
                               FROM $webcatiBaseTable._www_ids ids
                               LEFT JOIN $webcatiBaseTable._Projects p
                               ON ids.id_project = p.id
                               LEFT JOIN $webcatiBaseTable._wwwPointsStructure ps
                               ON ids.id_project = ps.id_project
                               WHERE user_type = 2 AND id_user = ? GROUP BY ids.id_project", [$userId]);

      $surveys = [
        /*"id" => "4325415241",
        "name" => "Ime in priimek",*/
        "surveys" => $mySurveys,
      ];

      /*$surveys = [
        "id" => "4325415241",
        "name" => "Ime in priimek",
        "surveys" => [
            [
                "ID" => "453453523626",
                "title" => "Raziskava uporabe beležk in kave",
                "shortDescription" => "Raziskava uporabe beležk in kave s krajšim opisom.",
                "duration" => "19 min",
                "targetGroup" => "Residents of Slovenia",
                "expires" => 1447068176,
                "points" => 2300,
                "image" => "http://url/do/slike.jpg"
            ],
            [
                "ID" => "453453523626",
                "title" => "Raziskava uporabe beležk in kaveaaa",
                "shortDescription" => "Raziskava uporabe beležk in kave s krajšim opisom.",
                "duration" => "20 min",
                "targetGroup" => "Residents of Slovenia",
                "expires" => 1447068176,
                "points" => 2300,
                "image" => "http://url/do/slike.jpg"
            ],
        ],
      ];*/

      return $surveys;
    }

    /*get percent of answered surveys*/
    public function getQuestionPosition($userId) {

        $webcatiBaseTable = env('TABLE_WARPIT_WEBCATI_BASE');

        $positions = array();

        $mySurveys = DB::select("SELECT ids.id_project, proj_name
                               FROM $webcatiBaseTable._www_ids ids
                               LEFT JOIN $webcatiBaseTable._Projects p
                               ON ids.id_project = p.id
                               LEFT JOIN $webcatiBaseTable._wwwPointsStructure ps
                               ON ids.id_project = ps.id_project
                               WHERE user_type = 2 AND id_user = ? GROUP BY ids.id_project", [$userId]);

        foreach($mySurveys as $index => $surveys_percent)
        {
            //npr. FROM statviewborut___enginelog project

            $tableNameEngineLog = $webcatiBaseTable . '.' . $surveys_percent->proj_name .'___EngineLog';

            $position = DB::select("SELECT id_structure, id_user, currentRotationString
                                   FROM $tableNameEngineLog
                                   WHERE id_user = ?
                                   ORDER BY id ASC", [$userId]);

            //if "status_rec" = 1, than survey is already successfuly answered..
            $tableNameStructure = $webcatiBaseTable . '.' . $surveys_percent->proj_name .'_structure';

            $status_rec = DB::selectOne("SELECT status_rec
                                         FROM $tableNameStructure
                                         WHERE user_id = ?", [$userId]);

            if($status_rec == 1) //survey answered
            {
                $positions[$index] = "100";
            }
            if($position == null) //if position not exists, than user stil didnt start answering..
            {
                $positions[$index] = "0";
            }
            elseif($status_rec!=1 && $position!=null) //survey is in the middle of answering..
            {
                $position_start = DB::selectOne("SELECT currentRotationString
                                   FROM $tableNameEngineLog
                                   WHERE id_user = ?
                                   ORDER BY id ASC", [$userId]);

                $position_end = DB::selectOne("SELECT currentRotationString
                                       FROM $tableNameEngineLog
                                       WHERE id_user = ?
                                       ORDER BY id DESC", [$userId]);

                $number_of_slash_start = substr_count($position_start, "/") -4; //-4 ... without first and last number qustion.. thats 4 "/"

                $all_questions = $number_of_slash_start + 1;

                $number_of_slash_end = substr_count($position_end, "/") -1; //-1 ... just to count numbers

                $percentLeft = $number_of_slash_end / $all_questions;

                //100 % - (percent left)
                $percForSolve = 1 - $percentLeft;

                $percForSolve = $percForSolve*100;

                $percForSolve = number_format((float)$percForSolve, 0, '.', ''); // round on 0 decimal places

                $positions[$index] = $percForSolve;
            }
        }

        $positions = [
        /*"id" => "4325415241",
        "name" => "Ime in priimek",*/
        "position" => $positions,
        ];


        return $positions;
     }

    //User activity by solving surveys
    public function surveysActivityOfUser($userId) {

        $webcatiBaseTable = env('TABLE_WARPIT_WEBCATI_BASE');

        $all_surveys = array();

        //First get all surveys from table _www_ids
        $tableIds =  $webcatiBaseTable . '._www_ids';
        $tableProjects =  $webcatiBaseTable . '._Projects';

        $all_surveys = DB::select("SELECT t.id_project, p.proj_name
                                   FROM $tableIds t
                                   LEFT JOIN $tableProjects p
                                   ON t.id_project = p.id
                                   WHERE t.id_user = ?", [$userId]);

        $number_of_all_surveys = count($all_surveys);

        //now we need count surveys that was been answeder 100%
        //Every survey have table "survey"_structure... inside that table we check if UserId exists and have "status_rec"=1 ..(answered survey)

        $number_of_answered_surveys = 0;

        foreach($all_surveys as $survey)
        {
            $tableStructure = $webcatiBaseTable . "." . $survey->proj_name . "_structure";

            $answered = DB::selectOne("SELECT s.status_rec
                                   FROM $tableStructure s
                                   WHERE s.user_id = ? AND s.status_rec=1", [$userId]);
            if($answered!=null)
                $number_of_answered_surveys++;

        }


        $surveysData = [
        /*"id" => "4325415241",
        "name" => "Ime in priimek",*/
        "number_of_all_surveys" => $number_of_all_surveys,
        "nubmer_of_answered_surveys" => $number_of_answered_surveys,
        ];


        return $surveysData;
    }

     //User activity by earning and spending points
    public function pointsActivityOfUser($userId,$filter) {

        $webcatiBaseTable = env('TABLE_WARPIT_WEBCATI_BASE');
        $all_activity = array();
        $table_wwwPoints = $webcatiBaseTable . '._wwwPoints';
        $days = 7;

        if(isset($filter)){
            if($filter == 1)
                $days = 7;
            if($filter == 2)
                $days = 30;
            if($filter == 3)
                $days = 365;
            if($filter == 4)
                $days == 99999;
        }
        $all_activity = DB::select("SELECT modificationDate, sum(case when webPoints > 0 then webPoints else 0 end) as positive, sum(case when webPoints < 0 then webPoints else 0 end) as negative
                                    FROM $table_wwwPoints
                                    WHERE id_user = ?
                                    AND DATE(FROM_UNIXTIME(`modificationDate`)) >= SUBDATE(DATE(NOW()), $days)
                                    GROUP BY DATE(FROM_UNIXTIME(modificationDate))", [$userId]);

        $activityPoints = [
        "all_activity" => $all_activity,
        ];


        return $activityPoints;
    }

    /**
     * Retrieves user points
     *
     * GET method for REST service - get current user points (still available). The data is calculated from table
     * _wwwPoints with reference to purpose, which define purposes of points.
     *
     * $params [int] userId
     */
    public function getUserPoints($userId)
    {
        $webcatiBaseTable = env('TABLE_WARPIT_WEBCATI_BASE');
        $currentTime = time();

        $sql = "SELECT SUM(points.points_available) AS points
            FROM `$webcatiBaseTable`.`_wwwPoints`  AS points
            JOIN `$webcatiBaseTable`.`_wwwPurpose` AS purpose
              ON points.id_purpose = purpose.id
            WHERE purpose.negative > 0
              AND points.timestamp_expiration > $currentTime
              AND points.deleted = 0
              AND points.id_user = ?";

        $points = DB::selectOne($sql, [$userId]);

        if ($points === null)
            $points = 0;

        $sql = "SELECT id_purpose AS purpose, webPoints as points, points_available,
            modificationDate, timestamp_expiration
            FROM `$webcatiBaseTable`.`_wwwPoints`
            WHERE id_user = ?
              AND deleted = 0
              AND timestamp_expiration >= $currentTime
            ORDER BY modificationDate DESC
        ";

        $transactions = DB::select($sql, [$userId]);

        return ['points' => $points, 'transactions' => $transactions];
    }

    /**
     * Add points to a user / subtract points from a user.
     *
     * Points can be positive or negative. Purpose is defined in the _wwwPurpose table.
     */
    public function addOrSubtractPoints($userId)
    {
        $surveyId = (int) $this->request->input('surveyId');
        $points   = (int) $this->request->input('points');
        $purpose  = (int) $this->request->input('purpose');

        if ($this->request->has('timestamp_expiration'))
            $timestampExpiration = $this->request->input('timestamp_expiration');
        else
            $timestampExpiration = null;

        if ($points >= 0) {
            $this->addTransaction($userId, $surveyId, $purpose, $points, $points, $timestampExpiration);
            return $this->getUserPoints($userId);
        }

        // In case points are negative, the procedure is a bit more involved.
        // We need to modify 'points_available' column from oldest non-expired transaction
        // forward until all points are accounted.

        DB::transaction(function() use ($userId, $points, $purpose) {
            $this->subtractPointsFromUser($userId, $points, $purpose);
        });

        return $this->getUserPoints($userId);
    }

    /**
     * Inserts a new row into _wwwPoints.
     */
    private function addTransaction($userId, $surveyId, $purpose, $points, $pointsAvailable, $timestampExpiration)
    {
        $webcatiBaseTable = env('TABLE_WARPIT_WEBCATI_BASE');
        $currentTime = time();

        $sql = "INSERT INTO `$webcatiBaseTable`.`_wwwPoints`
            (id_user, id_project, id_purpose,
             webPoints, points_available,
             modificationDate, timestamp_expiration,
             deleted, system)
             VALUES
            (?,?,?, ?,?, ?,?,?,?)";

        DB::insert($sql, [
            $userId, $surveyId, $purpose,
            $points, $pointsAvailable,
            $currentTime, $timestampExpiration,
            0, 0
        ]);
    }

    /**
     * Goes through a list of non-expired points from oldest to newest,
     * reduces points_available untils all $points are "subtracted".
     */
    private function subtractPointsFromUser($userId, $points, $purpose)
    {
        $webcatiBaseTable = env('TABLE_WARPIT_WEBCATI_BASE');
        $currentTime = time();

        $transactions = $this->getNonExpiredUserTransactions($userId);
        $remainingPoints = abs($points);

        foreach ($transactions as $transaction) {
            $transactionId = $transaction->id;
            $transactionPoints = (int) $transaction->points_available;
            if ($remainingPoints < $transactionPoints) {
                $transactionPoints -= $remainingPoints;
                $remainingPoints = 0;
            }
            else {
                $remainingPoints -= $transactionPoints;
                $transactionPoints = 0;
            }

            /*DB::update("
                UPDATE `$webcatiBaseTable`.`_wwwPoints`
                SET points_available = $transactionPoints,
                    modificationDate = $currentTime
                WHERE id = $transactionId
            ");*/
            //we cant modify modification date, becose we lose date, when user was earned points
            DB::update("
                UPDATE `$webcatiBaseTable`.`_wwwPoints`
                SET points_available = $transactionPoints
                WHERE id = $transactionId
            ");

            if ($remainingPoints == 0)
                break;
        }

        if ($remainingPoints != 0)
            throw new RuntimeException('Not enough points to spend!');

        // Record this transaction as well
        $this->addTransaction($userId, null, $purpose, $points, 0, null);
    }

    /**
     * Transfer points from one user to another.
     */
    public function transfer()
    {
        $from   = $this->request->input('from');
        $to     = $this->request->input('to');
        $points = $this->request->input('points');

        // Remove points from user $from (oldest non-expired to newest),
        // add them to user $to (expiration_date kept the same).

        DB::transaction(function () use ($from, $to, $points) {
            $webcatiBaseTable = env('TABLE_WARPIT_WEBCATI_BASE');
            $currentTime = time();
            $transactions = $this->getNonExpiredUserTransactions($from);
            $remainingPoints = abs($points);

            foreach ($transactions as $transaction) {
                $transactionId = $transaction->id;
                $transactionPoints = (int) $transaction->points_available;
                $pointsToTransfer = 0;
                if ($remainingPoints < $transactionPoints) {
                    $transactionPoints -= $remainingPoints;
                    $pointsToTransfer = $remainingPoints;
                    $remainingPoints = 0;
                }
                else {
                    $remainingPoints -= $transactionPoints;
                    $pointsToTransfer = $transactionPoints;
                    $transactionPoints = 0;
                }

                DB::update("
                    UPDATE `$webcatiBaseTable`.`_wwwPoints`
                    SET points_available = $transactionPoints,
                        modificationDate = $currentTime
                    WHERE id = $transactionId
                ");

                // Give these points to user $to.
                // Keep expiration date.
                $this->addTransaction($to, $surveyId=null, $purpose=1003, $pointsToTransfer, $pointsToTransfer,
                    $transaction->timestamp_expiration);

                if ($remainingPoints == 0)
                    break;
            }

            if ($remainingPoints != 0)
                throw new RuntimeException('Not enough points to transfer!');

            $this->addTransaction($from, null, $purpose=1002, -abs($points), 0, null);
        });
    }

    /**
     * Retrieves a list of transactions that are positive, still have points
     * available and are not expired.
     */
    private function getNonExpiredUserTransactions($userId)
    {
        $webcatiBaseTable = env('TABLE_WARPIT_WEBCATI_BASE');
        $currentTime = time();

        $transactions = DB::select("
            SELECT points.id, points.points_available, points.timestamp_expiration,
                   points.modificationDate
            FROM `$webcatiBaseTable`.`_wwwPoints`  AS points
            JOIN `$webcatiBaseTable`.`_wwwPurpose` AS purpose
              ON points.id_purpose = purpose.id
            WHERE points.timestamp_expiration > $currentTime
              AND points.id_user = ?
              AND points.deleted = 0
              AND points.points_available > 0
              AND purpose.negative > 0
            ORDER BY points.timestamp_expiration ASC
        ", [$userId]);

        return $transactions;
    }


    public function getUserPointsEarningLoss($userId){
        //earned points by me = registration points + survey answered + (orders)
        //earned points by friend
        //spend points
        //total points
        $webcatiBaseTable = env('TABLE_WARPIT_WEBCATI_BASE');
        $currentTime = time();

        $user_points = DB::select("SELECT id_purpose, label, SUM(webPoints) AS points
            FROM `$webcatiBaseTable`.`_wwwPoints`  AS points
            LEFT JOIN `$webcatiBaseTable`.`_wwwPurpose` AS purpose
                ON points.id_purpose = purpose.id
            WHERE points.id_user = ?
            GROUP BY id_purpose", [$userId]);

        $f_points = 0;
        $my_earning = 0;
        $my_loss = 0;

        foreach ($user_points as $value) {
           //friend points purpose (1005,1004,1003)
           if($value->id_purpose == 1005 || $value->id_purpose == 1004 || $value->id_purpose == 1003){
            $f_points = $value->points;
           }
           //my points purpose (registration, survey answered, recieved from admin)
           if($value->id_purpose == 1000 || $value->id_purpose == 1001 || $value->id_purpose == 1006){
            $my_earning = $value->points;
           }
           //spend points (send to friend, points abstracted from admin )
           if($value->id_purpose == 1002 || $value->id_purpose == 1007 || $value->id_purpose == 1008){
            $my_loss = $value->points;
           }

        }
        $up['earned_by_friend'] = $f_points;
        $up['earned_by_me'] = $my_earning;
        $up['spent'] = $my_loss;
        $up['total_points'] = $f_points+$my_earning;

        return $up;
    }
}
