<?php

require 'vendor/autoload.php';

use Ratchet\Server\IoServer;
use Ratchet\Http\HttpServer;
use Ratchet\WebSocket\WsServer;
use Ratchet\ConnectionInterface;
use Ratchet\MessageComponentInterface;

$sessionHandling = 1;

include_once('_dirinfo.php');
include_once(PATH_TO_ROOT . 'init.php');
define('SESSION_TIMEOUT', 6000);


class WarpitWebSocketServer implements MessageComponentInterface {

    private $_connections; // Websocket connections to browsers. For monitoring.

    private $publishers;  // SploObjectStorage of connections
    private $subscribers; // questionnaireId => SplObjectStorage of connections

    private $mysql; // MySQLi connection
    private $sessions = []; // Map session_id => last_impression_time


    public function __construct() {
        $this->_connections = new \SplObjectStorage;
        $this->publishers = new \SplObjectStorage;
        $this->subscribers = [];
    }


    public function setupDatabase() {
        // Setup the database connection.
        // TODO: setting timeout to 0 doesn't seem to work?
        $mysql = mysqli_init();
        $success = mysqli_options($mysql, MYSQLI_OPT_CONNECT_TIMEOUT, 0);
        if (! $success) {
            echo 'Failed to set mysql connection timeout to 0' . "\n";
            return null;
        }

        $success = mysqli_real_connect($mysql, WARPIT_HOST, MYSQL_USER, MYSQL_PASS);
        if (! $success) {
            echo "Failed to connect to mysql: " . mysqli_connect_error() . "\n";
            return null;
        }

        mysqli_query($mysql, "SET NAMES 'utf-8'");

        $this->mysql = $mysql;
        return $mysql;
    }


    private function queryWithReconnect($sql) {
        $result = mysqli_query($this->mysql, $sql);

        if ($result === false) {
            echo 'Mysql error: ' . mysqli_error($this->mysql) . "\n";
            echo 'Reestablishing the connection...' . "\n";
            $this->setupDatabase();

            $result = mysqli_query($this->mysql, $sql);
            if ($result === false) {
                echo 'Failed to reestablish the connection!' . "\n";
            }
        }

        return $result;
    }


    public function onOpen(ConnectionInterface $conn) {
        // Diagnostics.
        $pubCount = $this->publishers->count();
        $subCount = 0;
        foreach ($this->subscribers as $subs)
            $subCount += $subs->count();

        $allCount = $this->_connections->count();
        $questionnaireCount = count($this->subscribers);
        echo "Count (pub, sub, all, ques): $pubCount, $subCount, $allCount, $questionnaireCount\n";

        $this->_connections->attach($conn);
        echo "New connection: ({$conn->resourceId}).\n";
    }


    public function onMessage(ConnectionInterface $connection, $rawMessage) {
        // Main dispatch.
        // Messages are in JSON format.
        $message = json_decode($rawMessage, true);

        // Prevent non-authorized clients to post messages: check logged-in flag in Warpit session.
        // We use session cache in order not to query the database all the time.
        // We also check 'live_watch_subscriber_id' in session (needs
        // to match subscriber_id set in forms.admin.php) when subscribing
        // (only admins should be allowed to watch interviewees!).
        $sessionId = $message['session_id'];
        $checkSubscriberId = null;
        if ($message['type'] == 'subscribe')
            $checkSubscriberId = $message['subscriber_id'];

        if (! $this->checkSession($connection, $sessionId, $checkSubscriberId)) {
            // Not logged-in, terminate the connection immediately.
            $connection->close();
            return;
        }

        // Remove session_id from message (don't pass it to anyone).
        unset($message['session_id']);
        $rawMessage = json_encode($message);

        // Decide what to do.
        switch ($message['type']) {
            case 'subscribe':    $this->handleNewSubscriber($connection, $message);              break;
            case 'client_login': $this->handleClientLogin($connection,   $message, $rawMessage); break;
            case 'client_setup': $this->handleClientSetup($connection,   $message, $rawMessage); break;
            case 'client_event': $this->handleClientEvent($connection,   $message, $rawMessage); break;
            default:
                echo 'Received unkown message "' . substr($rawMessage, 0, 20) . '..." from ' .
                    $connection->resourceId . "\n";
        }
    }


    private function checkSession(ConnectionInterface $connection, $sessionId, $checkSubscriberId = null) {
        // Retrive Warpit/survey session from sessionId,
        // check logged_in flag and time since last impression.
        // Don't actually write anything to the database.

        // Check the session cache first: no need to query the database on every event.
        if (isset($this->sessions[$sessionId])) {
            $lastImpression = $this->sessions[$sessionId];
            $now = time(); // Second since Unix Epoch.
            if ($now - $lastImpression < SESSION_TIMEOUT)
                return true;
        }

        $time = -microtime(true); // Benchmark (since this is a synchronous operation in an asynchronous app)

        if (strpos($sessionId, '-') !== false) {
            // Survey session id.
            $expSID = explode('-', $sessionId, 2);
            if (! is_numeric($expSID[1])) {
                echo 'Invalid session_id: ' . $sessionId . "\n";
                return false;
            }

            // See _support/class.SurveySession.php
            $sql = "SELECT 1 AS logged_in,
                           variable_value                                  AS session_data,
                           TIME_TO_SEC(TIMEDIFF(NOW(), `last_impression`)) AS diff,
                           UNIX_TIMESTAMP(`last_impression`)               AS last_impression_unix
                    FROM " . DB_WARPIT_WEBCATI_BASE . "._www_interviewer_session
                    WHERE id = '{$expSID[1]}'";
        }
        else {
            $escapedSessionId = mysqli_real_escape_string($this->mysql, $sessionId);
            // See webcati/_class/class.UserSession.php
            $sql = "SELECT logged_in,
                           variable_value                                  AS session_data,
                           TIME_TO_SEC(TIMEDIFF(NOW(), `last_impression`)) AS diff,
                           UNIX_TIMESTAMP(`last_impression`)               AS last_impression_unix
                    FROM " . DB_WARPIT_MAIN . "._user_session AS a
                        LEFT JOIN " . DB_WARPIT_MAIN . "._session_variable AS b
                            ON a.id = b.id_session
                    WHERE id_ascii_session = '$escapedSessionId'";
        }

        $result = $this->queryWithReconnect($sql);

        if (! $result || mysqli_num_rows($result) !== 1) {
            if (! $result)
                $reason = '$result is false - ' . mysqli_error($this->mysql);
            else
                $reason = 'num_rows !== 1';
            echo 'Failed to find session_id: ' . $sessionId . ' (' . $reason . ")\n";
            return false;
        }

        $row = mysqli_fetch_assoc($result);
        $loggedIn = $row['logged_in'];
        $timeSinceImpression = $row['diff'];
        if ($timeSinceImpression > SESSION_TIMEOUT)
            $loggedIn = false;

        if (! $loggedIn) {
            echo 'session_id: ' . $sessionId . " not logged-in!\n";
            return false;
        }

        if ($checkSubscriberId !== null) {
            // We stored live_watch_subscriber_id in SESSION,
            // generated in forms.admin.php (uniqid()).
            // In order not to de-serialize the entire session,
            // let's extract just that variable.
            $serializedSession = $row['session_data'];
            $index = strpos($serializedSession, 'live_watch_subscriber_id');
            $start = strpos($serializedSession, '"', $index) + 1;
            $end = strpos($serializedSession, '"', $start);
            $lwsid = substr($serializedSession, $start, $end-$start);
            if ($lwsid != $checkSubscriberId) {
                echo 'SubscriberId check failed!' . "\n";
                return false;
            }
        }

        // Remember last impression time for this session id.
        $this->sessions[$sessionId] = $row['last_impression_unix'];

        echo "Query time: " . sprintf('%f', $time + microtime(true)) . "\n";
        return true;
    }


    private function handleNewSubscriber(ConnectionInterface $connection, $message) {
        $questionnaireId = $message['questionnaire_id'];
        $connection->questionnaireId = $questionnaireId;

        if (! isset($this->subscribers[$questionnaireId]))
            $this->subscribers[$questionnaireId] = new \SplObjectStorage;

        $this->subscribers[$questionnaireId]->attach($connection);

        echo "subscribe event handled\n";
    }


    private function handleClientLogin(ConnectionInterface $connection, $message, $rawMessage) {
        // Client just logged in, but has not yet selected a questionnaire.
        // Send this event to *all* subscribers.

        $clientId = $message['client_id'];
        $connection->questionnaireId = null; // Smuggle these values inside $connection object.
        $connection->clientId = $clientId;
        $this->publishers->attach($connection);

        $this->sendMessageToSubscribers($rawMessage);
    }


    private function handleClientEvent(ConnectionInterface $connection, $message, $rawMessage) {
        // A client sent an event (radio button selected etc), pass it on to all subscribers that are
        // listening on this questionnaire.
        //

        // TODO: this is the same as handleClientSetup?
        $questionnaireId = $message['questionnaire_id'];
        $clientId = $message['client_id'];
        $connection->questionnaireId = $questionnaireId;
        $connection->clientId = $clientId;
        $this->publishers->attach($connection); // Might already be attached, but that's ok (no duplicates).

        $this->sendMessageToSubscribers($rawMessage, $questionnaireId);
    }


    private function handleClientSetup(ConnectionInterface $connection, $message, $rawMessage) {
        // A client (interviewee) is trying to (re-)connect.
        // Notify all subscribers listening on this questionnaire that we have a new client!
        //

        $questionnaireId = $message['questionnaire_id'];
        $clientId = $message['client_id'];
        $connection->questionnaireId = $questionnaireId;
        $connection->clientId = $clientId;
        $this->publishers->attach($connection);

        $this->sendMessageToSubscribers($rawMessage, $questionnaireId);
    }


    private function sendMessageToSubscribers($rawMessage, $questionnaireId = null) {
        if ($questionnaireId === null) {
            // No questionnaireId specified, send to *all* subscribers.
            foreach ($this->subscribers as $questionnaireId => $subs)
                $this->sendMessageToSubscribers($rawMessage, $questionnaireId);

            return;
        }

        if (isset($this->subscribers[$questionnaireId])) {
            foreach ($this->subscribers[$questionnaireId] as $sub) {
                $sub->send($rawMessage);
            }
        }
    }


    public function onClose(ConnectionInterface $conn) {
        echo "Connection {$conn->resourceId} has disconnected\n";

        // If a client (interviewee) has disconnected, send
        // subscribers a notification.
        //
        // All other events are generated by publishers directly,
        // only this one is synthetic.
        //
        // TODO: this assumes connection is closed properly, do we
        // need to do ping-pong?
        // - TCP keepalive
        // - WebSockets can have ping-pong (but no JavaScript API)
        // - https://github.com/ratchetphp/Ratchet/commit/6da3b04e2d2cb1fb04fd6e3288ed8c2942ba4f94
        // - http://stackoverflow.com/questions/26355077/overhead-of-idle-websockets?rq=1
        $questionnaireId = $conn->questionnaireId;
        $subs = $this->subscribers[$questionnaireId];
        if ($questionnaireId != null && isset($subs) && $subs->contains($conn)) {
            // Remove from subscribers.
            $questionnaireId = $questionnaireId;
            $subs->detach($conn);
        }
        else {
            // A publisher. Might not have questionnaireId yet (login page).
            $message = json_encode([
                'type' => 'client_closed',
                'client_id' => $conn->clientId,
                'questionnaire_id' => $conn->questionnaireId
            ]);

            $this->sendMessageToSubscribers($message);

            $this->publishers->detach($conn); // Might not be in here, but that is ok.
        }

        $this->_connections->detach($conn);
    }


    public function onError(ConnectionInterface $conn, \Exception $e) {
        echo "An error has occurred: {$e->getMessage()}\n";
        $conn->close();
    }
}



function main() {
    echo "\n" . 'Starting ' . date('Y-m-d H:i:s') . "\n";
    echo 'Running as uid ' . posix_getuid() . ' and gid ' . posix_getgid() . "\n";

    $ws = new WarpitWebSocketServer();
    $mysql = $ws->setupDatabase();
    if ($mysql === null)
        return;

    // Retrive path for log and pid files.
    $sql = 'SELECT cValue AS tmp_path
            FROM  ' . DB_WARPIT_MAIN . '._WarpitConfig
            WHERE prefix = "live_watch"
              AND cName = "tmp_path"';
    $result = mysqli_query($mysql, $sql);
    if (! $result || mysqli_num_rows($result) !== 1) {
        echo 'Failed to retrieve tmp path!' . "\n";
        return;
    }

    $row = mysqli_fetch_assoc($result);
    $tmpPath = $row['tmp_path'];
    $pidFile = $tmpPath . 'pid';

    // Save process id into pidfile. When this server process
    // is stoped, remove it.
    file_put_contents($pidFile, posix_getpid());

    // Quit normally (and call registered shutdown function when a SIGINT or SIGTERM is received).
    //
    // pcntl_signal seems to interfere with Ratchet, let's disable this.
    /*
    pcntl_signal(SIGINT, function() { exit; });
    pcntl_signal(SIGTERM, function() { exit; });

    // TODO: does not get called by kill -SIGINT or -SIGTERM ...
    // But we handle the case where pid file exists even though the process
    // is dead just fine (forms.admin.php).
    register_shutdown_function(function() use($pidFile) {
        echo 'Removing pid file...' . "\n";
        unlink($pidFile);
    });
    */

    $server = IoServer::factory(
        new HttpServer(
            new WsServer(
                $ws
            )
        ),
        8000
    );

    echo 'Server up!' . "\n";
    $server->run();
}


main();

?>
