<?php

class DBTest extends PHPUnit_Framework_TestCase
{
    public function setUp()
    {
        $this->testTableName = 'test_table_' . time();
        DB::query(
            "CREATE TABLE {$this->testTableName} (
                id int(11) AUTO_INCREMENT,
                str1 varchar(255),
                str2 varchar(255),
                PRIMARY KEY(id)
            ) ENGINE=InnoDB"
        );
    }


    public function tearDown()
    {
        DB::query("DROP TABLE IF EXISTS {$this->testTableName}");
    }


    public function testSelectInsertQuery()
    {
        $t = $this->testTableName;
        DB::insert("INSERT INTO $t VALUES (1, :str1, :str2)",    ['str1' => 'howdy1', 'str2' => 'world1']);
        DB::insert("INSERT INTO $t VALUES (null, :str1, :str2)", ['str1' => 'howdy2', 'str2' => 'world2']);
        DB::insert("INSERT INTO $t VALUES ?", [[null, 0, 'world3']]);

        $obj = DB::selectOne("SELECT str1, str2 AS renamed_column FROM $t WHERE id = :id", [':id' => 1]);

        $this->assertEquals('howdy1', $obj->str1);
        $this->assertEquals('world1', $obj->renamed_column);

        $str1 = DB::selectOne("SELECT str1 FROM $t WHERE id = 4242");
        $this->assertNull($str1);

        $count = DB::selectOne("SELECT count(*) FROM $t");
        $this->assertEquals(3, $count);
    }


    public function testAffected()
    {
        $t = $this->testTableName;
        $numAffected = DB::insert("INSERT INTO $t VALUE (420, 'value', 'string')");
        $this->assertEquals(1, $numAffected);

        $numAffected = DB::delete("DELETE FROM $t WHERE id = 420");
        $this->assertEquals(1, $numAffected);
    }


    public function testSelectOne()
    {
        $t = $this->testTableName;
        DB::insert("INSERT INTO $t VALUE (420, 'value', 'string')");

        $str1 = DB::selectOne("SELECT str1 FROM $t WHERE str1 = :str1", [':str1' => 'does not exist']);
        $this->assertNull($str1);
    }


    public function testTransactions()
    {
        $t = $this->testTableName;

        try {
            DB::transaction(function() use($t) {
                DB::insert("INSERT INTO $t VALUES (null, 'one', 'two')");
                DB::insert("INSERT INTO $t VALUES (null, :str1, :str2)", ['str1' => 'howdy1', 'str2' => 'world']);
                DB::insert("INSERT INTO $t VALUES (null, ?, ?)", ['test', 'another']);

                trigger_error('Forcing rollback.', E_USER_NOTICE);
            });
        }
        catch (ErrorException $e) {
            if (strpos($e->getMessage(), 'Forcing rollback.') === false)
                throw $e;
        }

        // Transaction must be rolled back.
        $count = DB::selectOne("SELECT count(*) FROM $t");
        $this->assertEquals(0, $count);

        // Now test commit.
        DB::transaction(function() use($t) {
            DB::insert("INSERT INTO $t VALUES (null, null, null)");
        });

        $count = DB::selectOne("SELECT count(*) FROM $t");
        $this->assertEquals(1, $count);
    }
}
