<?php

namespace SolveX\Session;

/**
 * SessionLockingProxy wraps backend session handlers (e.g. database or redis),
 * preventing simultaneous access (e.g. multiple almost synchronous AJAX calls).
 *
 * This is done similarly to PHP's default file session handler (using flock).
 */
class SessionLockingProxy extends SessionProxy
{
    private $f;
    private $sessionName;
    private $lockDirectory;


    public function __construct($handler, $lockDirectory)
    {
        $this->handler = $handler;
        $this->lockDirectory = $lockDirectory;
    }


    public function open($savePath, $sessionName)
    {
        $this->sessionName = $sessionName;
        return $this->handler->open($savePath, $sessionName);
    }


    private function getLockFilePath($sessionId)
    {
        return $this->lockDirectory . DIRECTORY_SEPARATOR .
               $this->sessionName . '_' . $sessionId . '.lock';
    }


    public function read($sessionId)
    {
        // Create a lock file for this session.
        // flock() blocks if this file is already locked by another process.
        $this->f = fopen($this->getLockFilePath($sessionId), 'w+');
        flock($this->f, LOCK_EX);

        return $this->handler->read($sessionId);
    }


    public function write($sessionId, $data)
    {
        $result = $this->handler->write($sessionId, $data);

        // Unlock the lock file.
        flock($this->f, LOCK_UN);
        fclose($this->f);
        // TODO: unlinking the lock file here
        // is not a good idea because locking doesn't work anymore in that case.
        // But we need to eventually cleanup these lock files!
        // Maybe wrap gc()?

        return $result;
    }
}
