<?php

namespace SolveX\Session;

use DB;

/**
 * Session management.
 *
 */
class Session
{
    protected $sessionDriver = null;


    public function __construct()
    {
        // https://gist.github.com/eddmann/10262795
        // http://eddmann.com/posts/securing-sessions-in-php/
        ini_set('session.use_cookies', 1);
        ini_set('session.use_only_cookies', 1);

        session_name(env('SESSION_NAME'));

        // https://github.com/ezimuel/PHP-Secure-Session/blob/master/SecureSession.php
        session_set_cookie_params(
            $lifetime = 0,
            $path     = ini_get('session.cookie_path'),
            $domain   = ini_get('session.cookie_domain'),
            $secure   = isset($_SERVER['HTTPS']),
            $httponly = true
        );

        $sessionDriver = env('SESSION_DRIVER');

        if ($sessionDriver === 'redis')
            $handler = new RedisSessionHandler();
        else if ($sessionDriver === 'db')
            $handler = new DatabaseSessionHandler();
        else
            throw new \RuntimeException('Unknown SESSION_DRIVER!');

        $this->sessionDriver = $handler;

        // Use session locking if requested.
        $sessionLockPath = env('SESSION_LOCK_PATH', '');
        if ($sessionLockPath)
            $handler = new SessionLockingProxy($handler, $sessionLockPath);

        $success = session_set_save_handler(
            [$handler, 'open'],
            [$handler, 'close'],
            [$handler, 'read'],
            [$handler, 'write'],
            [$handler, 'destroy'],
            [$handler, 'gc']
        );

        if (! $success)
            throw new \RuntimeException('Failed to set session handler!');

        // As recommended by the docs,
        // the following prevents unexpected effects when using objects as save handlers.
        // http://php.net/manual/en/function.session-set-save-handler.php
        register_shutdown_function('session_write_close');

        session_start();

        // SessionHandler->open() and read() methods are called

        $this->checkLastActivity();
    }


    /**
     * Retrieve a value from the session.
     *
     * @param string $key     Key of the value to retrieve.
     * @param mixed  $default If $key is not in session, return $default.
     *
     * @return mixed
     */
    public function get($key, $default = null)
    {
        if (! isset($_SESSION[$key]))
            return $default;

        return $_SESSION[$key];
    }


    /**
     * Store a value in the sesion.
     *
     */
    public function put($key, $value)
    {
        $_SESSION[$key] = $value;
    }


    /**
     * Retrieve a value from the session and then remove it.
     *
     */
    public function pull($key)
    {
        $value = $_SESSION[$key];

        unset($_SESSION[$key]);

        return $value;
    }


    /**
     * Does sesion have a value under this key?
     *
     * @return bool
     */
    public function has($key)
    {
        return isset($_SESSION[$key]);
    }


    /**
     * Empty the session.
     *
     */
    public function flush()
    {
        $_SESSION = [];
    }


    /**
     * Retrieve session id.
     *
     * @return string
     */
    public function id()
    {
        return session_id();
    }


    /**
     * Regenerate session id. If passed true, also destroys backend session
     * data for previous session id ($_SESSION is kept intact).
     *
     * @param bool $destroy Destroy session data (on disk/in database/memory).
     *
     */
    public function regenerate($destroy = true)
    {
        $sessionCopy = $_SESSION;

        // Create a new session id.
        // If $destroy is true, also delete the previous session data.
        session_regenerate_id($destroy);

        $_SESSION = $sessionCopy;
    }


    /**
     * Write session data to the backend and close it.
     *
     */
    public function close()
    {
        // Release possible locks.
        session_write_close();
    }


    /**
     * @deprecated
     *
     * Validate user credentials and login the user if everything is ok.
     *
     * <code>
     * $success = Session::login(['username' => 'Me', 'password' => '123456']);
     * if ($success) {}
     * </code>
     *
     * Instead of 'username' one can also use 'email', etc.
     * Additional fields also possible, e.g. 'deleted_at' => null to
     * check user is not deleted.
     *
     * On successful password verification, 'logged_in' session variable
     * is set to true and session id is regenerated (session fixation/hijacking issues).
     *
     * This function retrieves a row from DB_USER_TABLE.
     * Uses password_verify (PHP >= 5.5).
     *
     * @return bool
     */
    public function login($credentials)
    {
        if (count($credentials) < 2)
            throw new \InvalidArgumentException(
                'Login credentials must include at least "password" and a user identifier!');

        if (! isset($credentials['password']))
            throw new \InvalidArgumentException('Login credentials require password!');

        $password = $credentials['password'];

        $user = $this->findUserFromCredentials($credentials);

        // Note: $user->password is actually the hash of the password
        if (! $user || ! $user->password || ! password_verify($password, $user->password))
            return false;

        if (env('SESSION_ONE_LOGIN_ONLY', false) &&
            env('SESSION_DRIVER') === 'db')
            $this->sessionDriver->removeSessionsOf($user->id); // Remove previous sessions of this user

        $this->regenerate();
        $this->put('logged_in', true);
        $this->put('user_id', $user->id);
        $this->generateCsrfToken();

        return true;
    }


    protected function findUserFromCredentials($credentials)
    {
        $userTable = env('DB_USER_TABLE');
        unset($credentials['password']);

        $sql = 'SELECT id, password FROM `' . $userTable . '` WHERE ';
        $where = [];
        $bindings = [];
        foreach ($credentials as $attr => $value) {
            $where[] = "`$attr` = ?";
            $bindings[] = $value;
        }

        $sql .= implode(' AND ', $where);
        $user = DB::selectOne($sql, $bindings);
        return $user;
    }


    /**
     * @deprecated
     *
     * Generates a CSRF token and sends it in the 'csrftoken' cookie.
     * This method is automatically called when user logins!
     *
     * @link http://php.net/manual/en/function.hash.php
     */
    public function generateCsrfToken()
    {
        // http://stackoverflow.com/questions/6287903/how-to-properly-add-csrf-token-using-php
        $token = bin2hex(mcrypt_create_iv(32, MCRYPT_DEV_URANDOM));

        setcookie('csrftoken', $token, 0, '/');

        // All POST forms must include a hidden token field.
        // Storing csrftoken in session for easy access while
        // generating these forms.
        $this->put('csrftoken', $token);
    }


    /**
     * @deprecated
     *
     * Determines whether or not 'logged_in' session variable is true.
     *
     */
    public function isLoggedIn()
    {
        return $this->get('logged_in', false);
    }


    /**
     * @deprecated
     *
     * Sets 'logged_in' session variable to false, everything
     * else is flushed.
     *
     */
    public function logout()
    {
        $this->flush();
        $this->put('logged_in', false);
    }


    protected function checkLastActivity()
    {
        $currentTime = time(); // Number of seconds since the Unix epoch, timezone independent.

        if (! $this->has('last_activity')) {
            // Empty session (either new or session timed-out).
            // Let's add 'last_activity' variable, and
            // set 'logged_in' to false.
            $this->logout();
            $this->put('last_activity', $currentTime);
        }

        // Refresh 'last_activity'.
        // Check for session timeout.
        $lastActivity = $this->get('last_activity');

        $this->put('last_activity', $currentTime);

        if (($currentTime - $lastActivity) > env('SESSION_TIMEOUT')) {
            // Empty the session. Set 'logged_in' to false.
            $this->logout();
        }
    }
}
