<?php

namespace SolveX\Session;

use DB;
use Session as SessionFacade;

/**
 * Database session backend.
 *
 * Database schema:
 *
 * SESSION_DB_TABLE env configuration.
 *
 * e.g.:
 *
 * CREATE TABLE `session` (
 * `session_id` char(36) NOT NULL,
 * `session_data` mediumtext
 * ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
 *
 * ALTER TABLE `session`
 * ADD PRIMARY KEY (`session_id`);
 *
 *
 * Note: Code assumes session_id is UNIQUE (and *only* session_id column is unique!).
 *
 */
class DatabaseSessionHandler
{

    public function open($savePath, $sessionName)
    {
        return true;
    }


    public function close()
    {
        return true;
    }


    public function read($sessionId)
    {
        $sessionTable = env('SESSION_DB_TABLE');
        $sessionData = DB::selectOne(
            "SELECT session_data AS data FROM `$sessionTable` WHERE session_id = ?",
            [$sessionId]
        );

        if (! $sessionData)
            return '';

        return $sessionData;
    }


    public function write($sessionId, $data)
    {
        $sessionTable = env('SESSION_DB_TABLE');

        // INSERT or UPDATE
        // Assumes session_id is UNIQUE (and only session_id column is unique!).
        //
        $success = DB::insert(
            "INSERT INTO `$sessionTable` (session_id, session_data) VALUES (?, ?)
             ON DUPLICATE KEY UPDATE
               session_id = VALUES(session_id),
             session_data = VALUES(session_data)",
            [$sessionId, $data]
        );

        // We might want user_id or logged_in values replicated
        // in separate columns for faster SQL access in analytics etc...
        if ($success && env('SESSION_DB_COPY', ''))
            $this->copyToSeparateColumns();

        return $success;
    }


    private function copyToSeparateColumns()
    {
        $sessionTable = env('SESSION_DB_TABLE');
        $columns = env('SESSION_DB_COPY');
        $columns = explode(',', $columns);
        if (count($columns) < 1)
            return;

        // Get requested columns from session data.
        $mapToValue = function($column) {
            return SessionFacade::get($column, 0);
        };
        $values = array_map($mapToValue, $columns);
        $values[] = SessionFacade::id();

        // e.g. SET user_id = ?, logged_in = ?
        $mapToSql = function($column) {
            return "`$column` = ?";
        };
        $columns = array_map($mapToSql, $columns);
        $columns = implode(',', $columns);

        DB::update(
            "UPDATE `$sessionTable` SET $columns WHERE session_id = ?",
            $values
        );
    }


    /**
     * Removes all session data of $userId from session database table.
     * This is used e.g. when $userId logs in - we want only one session for each user.
     *
     * @todo: Think about using $handler->destroy($sessionId)
     */
    public function removeSessionsOf($userId)
    {
        $sessionTable = env('SESSION_DB_TABLE');
        $columns = env('SESSION_DB_COPY', '');
        $columns = explode(',', $columns);
        if (! in_array('user_id', $columns))
            return;

        DB::delete("DELETE FROM `$sessionTable` WHERE user_id = ?", [$userId]);
    }


    public function destroy($sessionId)
    {
        $sessionTable = env('SESSION_DB_TABLE');

        $rows = DB::delete(
            "DELETE FROM $sessionTable WHERE session_id = :id",
            ['id' => $sessionId]
        );

        return ($rows == 1);
    }


    public function gc($maxlifetime)
    {
        return true;
    }
}
