<?php

namespace SolveX\Routing;

use SolveX\Exceptions\NotFoundHttpException;
use SolveX\Interfaces\RouteInterface;
use SolveX\Interfaces\RequestInterface;
use Illuminate\Contracts\Container\Container as ContainerInterface;

/**
 * Routing.
 *
 */
class Route implements RouteInterface
{
    /**
     * Registered routes.
     */
    protected $routes = [];

    protected $request;

    protected $container;

    public function __construct(RequestInterface $request, ContainerInterface $container)
    {
        $this->request = $request;
        $this->container = $container;
    }

    /**
     * Register a Route callback (GET, POST, PUT, DELETE).
     *
     * <code>
     *
     * Route::get('/users/{id}', function($userId) {
     *     $user = DB::selectOne('SELECT * FROM users WHERE id = ?', [$userId]);
     *
     *     return $user; // Automatically converts to JSON
     * });
     *
     * Route::post('/users', function() {
     *     // Create a new user...
     * });
     *
     * Route::delete('/users/{id}', 'authorize', function($userId) {
     *     // Middleware 'authorize' is just a function. 'authorize' receives the
     *     // same parameters $request and $userId as this route callback.
     *     // Middleware can abort a respose early if authorization fails or many other things.
     *
     *     // Delete $userId from users table...
     * });
     *
     * </code>
     */
    protected function add($method, $arguments)
    {
        $path = $arguments[0];

        array_shift($arguments);
        $middlewareAndRoute = $arguments;

        $this->routes[] = [
            'method' => $method,
            'path' => $path,
            'middlewareAndRoute' => $middlewareAndRoute,
        ];
    }


    public function get()
    {
        $this->add('GET', func_get_args());
    }


    public function post()
    {
        $this->add('POST', func_get_args());
    }


    public function put()
    {
        $this->add('PUT', func_get_args());
    }


    public function delete()
    {
        $this->add('DELETE', func_get_args());
    }


    /**
     * Actually perform action registered with routes, subject
     * to incoming request.
     */
    public function dispatch()
    {
        $routes = $this->routes;
        $dispatcher = \FastRoute\simpleDispatcher(function(\FastRoute\RouteCollector $routeCollector) use($routes) {
            foreach ($routes as $route)
                $routeCollector->addRoute($route['method'], $route['path'], $route);
        });

        $request = $this->request;
        $routeInfo = $dispatcher->dispatch($request->method(), $request->path());
        if ($routeInfo[0] === \FastRoute\Dispatcher::NOT_FOUND ||
            $routeInfo[0] === \FastRoute\Dispatcher::METHOD_NOT_ALLOWED)
            throw new NotFoundHttpException; // Check your routes.php!

        $route = $routeInfo[1];
        $parameters = $routeInfo[2];
        $this->callMiddlewareAndRoute($route, $parameters);
    }


    /**
     * Route matched, time for action.
     */
    protected function callMiddlewareAndRoute($route, $parameters)
    {
        $middlewareAndRoute = $route['middlewareAndRoute'];

        foreach ($middlewareAndRoute as $callable)
            $result = $this->container->call($callable, $parameters);

        $this->processRouteResult($result);
    }


    /**
     * Sets response headers and outputs content.
     *
     * If you don't want this functionality, you can still call exit()
     * in your route callback at any time.
     *
     * <code>
     *
     * Route::get('/users/{id}', function($userId) {
     *     $user = DB::selectOne('SELECT * FROM users WHERE id = ?', [$userId]);
     *
     *     return $user; // Automatically converts to JSON.
     *                   // Returning string would set content-type to HTML.
     * });
     *
     * </code>
     */
    protected function processRouteResult($result)
    {
        $inTesting = php_sapi_name() === 'cli';

        if (! $inTesting) {
            header('Cache-Control: no-cache, must-revalidate');
            header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');
            header('Content-type: text/html; charset=utf-8');
            http_response_code(200);
        }

        if (is_array($result) || is_object($result)) {
            header('Content-Type: application/json');
            echo(json_encode($result));
        }
        else if (is_string($result))
            echo($result);
        else if (is_int($result))
            http_response_code($result);

        if (! $inTesting) // phpunit tests: must not call exit()
            exit();
    }
}
