<?php

namespace SolveX\Log;

use SolveX\Interfaces\LogInterface;

/**
 * Simple logging.
 *
 * Configuration options: LOG_FILE.
 */
class Log implements LogInterface
{
    protected $logLevel;

    public function __construct()
    {
        $logLevel = env('LOG_LEVEL', 'none');

        switch ($logLevel) {
        case 'debug': $logLevel = 1; break;
        case 'info':  $logLevel = 2; break;
        case 'error': $logLevel = 3; break;
        case 'none':  $logLevel = 4; break; // Don't log.
        default:
            throw new \RuntimeException('Unknown LOG_LEVEL!');
        };

        $this->logLevel = $logLevel;
    }


    /**
     * Log an error. RFC 5424: Runtime errors that do not
     * require immediate action but should typically be logged and monitored.
     *
     * @param string $message
     * @param array $context Additional information you want to add.
     *
     */
    public function error($message, $context = [])
    {
        $this->process(3, $message, $context);
    }


    /**
     * Log on info level. RFC 5424: Interesting events. Examples: User logs in, SQL logs.
     *
     * @param string $message
     * @param array $context Additional information you want to add.
     *
     */
    public function info($message, $context = [])
    {
        $this->process(2, $message, $context);
    }


    /**
     * Log on debug level. RFC 5424: Detailed debug information.
     *
     * @param string $message
     * @param array $context Additional information you want to add.
     *
     */
    public function debug($message, $context = [])
    {
        $this->process(1, $message, $context);
    }


    /**
     * Checks LOG_LEVEL, outputs log $message together with
     * $context (JSON encoded, entries separated with "\n" newline) into LOG_FILE.
     * Each log entry includes:
     *  - log level
     *  - time
     *  - remote IP (if set)
     */
    protected function process($level, $message, $context)
    {
        if ($level < $this->logLevel)
            return;

        $file = env('LOG_FILE', '');
        if (! is_writable($file))
            return;

        switch ($level) {
        case 1: $level = 'debug'; break;
        case 2: $level = 'info'; break;
        case 3: $level = 'error'; break;
        default: $level = '?'; break;
        };

        $remoteIp = isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '?';
        $payload = [
            'level' => $level,
            'time' => date('Y-m-d H:i:s'),
            'remote_ip' => $remoteIp,
            'message' => $message
        ];
        $payload += $context;
        $payload = json_encode($payload);
        $payload = $payload . PHP_EOL;

        file_put_contents($file, $payload, FILE_APPEND);
    }
}
