<?php

namespace SolveX\Database;

use SolveX\Database\MysqlFetchIterator;
use SolveX\Interfaces\DatabaseInterface;

class PDO implements DatabaseInterface
{
    protected $pdo = null;
    protected $pdo_options = array(
        \PDO::ATTR_ERRMODE               => \PDO::ERRMODE_EXCEPTION);

    protected $fetchStyle = \PDO::FETCH_OBJ;

    private $paramIndex = 1;



    public function __construct($config)
    {
        $host     = $config['host'];
        $user     = $config['user'];
        $password = $config['pass'];
        $database = $config['database'];

        $this->pdo_options[\PDO::ATTR_DEFAULT_FETCH_MODE] = $this->fetchStyle;

        try{
            $this->pdo = new \PDO("mysql:host=".$host.";dbname=".$database.";charset=utf8", $user, $password, $this->pdo_options);
        }catch(\PDOException $e){
            throw new \RuntimeException("PDO connection error" . $e->getMessage());
        }
    }


    /**
     * Run a select statement and return a single result.
     *
     * Example:
     * <code>
     * $id = $_POST['id']; // No need for escaping, it's done by DB.
     * $user = DB::selectOne('SELECT age, name FROM user WHERE id = ?', [$id]);
     * // Instead of question mark placeholders you can also use named placeholders:
     * $user = DB::selectOne('SELECT age, name FROM user WHERE id = :id', ['id' => $id]);
     * $newAge = $user->age + 1;
     * </code>
     *
     * If retrieving one field only, selectOne() returns just that field value, not a row object:
     * <code>
     * $age = DB::selectOne('SELECT age FROM user WHERE id = ?', [$id]);
     * </code>
     *
     * @param string $sql Query to run.
     * @param array $bindings Bindings.
     *
     * @return mixed An instance of stdClass class, string, or null if no result.
     *
     * @link http://jpauli.github.io/2014/07/21/php-and-mysql-communication-mysqlnd.html
     * @link http://blog.ulf-wendel.de/2014/php-5-7-mysqlnd-memory-optimizations/
     *
     */
    public function selectOne($sql, $bindings = [])
    {
        $statement = $this->prepareSQL($sql, $bindings);

        $statement->execute();

        if($statement->columnCount() == 1)
            $obj = $statement->fetch(\PDO::FETCH_NUM)[0];
        else
            $obj = $statement->fetch();

        $statement->closeCursor();

        return $obj;
    }


    /**
     * Run a select statement against the database. Returns an array of objects.
     *
     * Example:
     * <code>
     * $age = $_GET['age']; // No need for escaping, it's done by DB.
     * $users = DB::select('SELECT id, name FROM user WHERE age < :age', ['age' => $age]);
     * foreach ($users as $user) {
     *     echo $user->id;
     * }
     * </code>
     *
     * If retrieving one field only (column), select() returns just that field values, not row objects:
     * <code>
     * $ages = DB::select('SELECT age FROM user WHERE parent = ?', [$parentId]);
     * // e.g. $ages = ['12', '42']
     * </code>
     *
     * @param  string  $sql       Query to run.
     * @param  array   $bindings  Bindings.
     * @return array              Array of stdClass instances / strings.
     *
     * @link http://jpauli.github.io/2014/07/21/php-and-mysql-communication-mysqlnd.html
     * @link http://blog.ulf-wendel.de/2014/php-5-7-mysqlnd-memory-optimizations/
     */
    public function select($sql, $bindings = [])
    {
        $statement = $this->prepareSQL($sql, $bindings);

        $statement->execute();

        $resultset = array();

        if($statement->columnCount() == 1){
            foreach ($statement->fetchAll(\PDO::FETCH_NUM) as $key => $value) {
                $resultset[] = $value[0];
            }
        }
        else{
            foreach ($statement->fetchAll() as $key => $value) {
                $resultset[] = $value;
            }
        }

        $statement->closeCursor();

        return $resultset;
    }


    /**
     * An insert statement.
     *
     * Example of an INSERT query. Notice NOW() is in the query itself,
     * not the bindings.
     * <code>
     * DB::insert('INSERT INTO db2.log(datetime, type, severity, message)
     *             VALUES (NOW(), :type, :severity, :message)', [
     *                 'type' => 2,
     *                 'severity' => 3,
     *                 'message' => 'User age updated'
     *             ]);
     * </code>
     *
     * @param  string   $sql        Query to run.
     * @param  array    $bindings   Bindings.
     * @return bool                 Success/failure.
     */
    public function insert($sql, $bindings = [])
    {
        $statement = $this->prepareSQL($sql, $bindings);

        return $statement->execute();
    }


    /**
     * An update statement.
     *
     * @param  string   $sql       Query to run.
     * @param  array    $bindings  Bindings.
     * @return int                 Num of affected rows.
     */
    public function update($sql, $bindings = [])
    {
        return $this->affectingStatement($sql, $bindings);
    }


    /**
     * A delete statement.
     *
     * @param  string  $sql       Query to run.
     * @param  array   $bindings  Bindings.
     * @return int                Num of affected rows.
     */
    public function delete($sql, $bindings = [])
    {
        return $this->affectingStatement($sql, $bindings);
    }


    /**
     * For update or delete statements, we want to get the number of rows affected
     * by the statement and return that back.
     */
    protected function affectingStatement($sql, $bindings)
    {
        $statement = $this->prepareSQL($sql, $bindings);

        $statement->execute();
        return $statement->rowCount();
    }


    /**
     * Plain query.
     *
     * @param string $sql
     * @param array $bindings
     *
     * @return mixed mysqli_query return value (depends on the query itself).
     */
    public function query($sql, $bindings = [])
    {
        $statement = $this->prepareSQL($sql, $bindings);
        return $statement->execute();
    }


    /**
     * Runs database queries in a transaction.
     *
     * Any exception in the closure rollbacks the transaction. Otherwise the transaction
     * is automatically committed.
     * Exceptions thrown are re-thrown after the rollback so that they can be handled
     * how the developer sees fit.
     *
     * Example:
     * <code>
     * DB::transaction(function() {
     *     DB::update('UPDATE account SET balance = balance - ? WHERE id = ?', [500, 12]);
     *     DB::update('UPDATE account SET balance = balance + ? WHERE id = ?', [500, 42]);
     * });
     * </code>
     *
     * Another example with locking:
     * <code>
     * DB::transaction(function() {
     *     $room = DB::selectOne('SELECT id, status FROM room WHERE empty = 1 FOR UPDATE');
     *     // SELECT FOR UPDATE reads the latest available data, setting exclusive locks
     *     // on each row it reads. Only applies in a transaction.
     *     DB::update('UPDATE room SET status = 2 WHERE id = ?', [$room->id]);
     * });
     * </code>
     *
     * Transactions are not available on MyISAM tables (use InnoDB)!
     *
     * Some statements cannot be rolled back (they implicitly commit). In general, these include data definition
     * language (DDL) statements, such as those that create or drop databases,
     * those that create, drop, or alter tables or stored routines.
     *
     * When the script ends or when a connection is about to be closed,
     * if you have an outstanding transaction, it will automatically rolled back.
     * This is a safety measure to help avoid inconsistency in the cases
     * where the script terminates unexpectedly--if you didn't explicitly
     * commit the transaction, then it is assumed that something went awry,
     * so the rollback is performed for the safety of your data.
     *
     * Nested transactions are not supported.
     *
     * MySQL's default transaction isolation level is REPEATABLE READ.
     * This means updates/inserts/deletes (even committed) from concurrent transactions
     * are not visible to this transaction.
     * For protection from concurrent transactions,
     * consider Locking reads (e.g. SELECT FOR UPDATE).
     * http://dev.mysql.com/doc/refman/5.7/en/innodb-locking-reads.html
     * http://dba.stackexchange.com/a/15864
     * http://stackoverflow.com/questions/10935850/when-to-use-select-for-update
     *
     * @param  Closure  $closure  Closure to run in a transaction.
     *
     * @return mixed Return value of your $closure.
     */
    public function transaction(\Closure $closure)
    {
        $exceptionOrNull = null;

        $this->pdo->beginTransaction();

        try{
            $result = $closure();
            $this->pdo->commit();
        }
        catch(\Exception $e){
            $this->pdo->rollBack();
            $exceptionOrNull = $e;
        }

        if($exceptionOrNull)
            throw $exceptionOrNull;

        return $result;
    }

    /**
     * Prapres myqsl query into a prepared statment and binds parameters
     * @param  string       $sql        MySQL string
     * @param  array        $bindings   Array of needed parameters
     * @return PDOStatement             returns a PDOStatment that is read to be executed
     */
    protected function prepareSQL($sql, $bindings){
        if(empty($bindings))
            return $this->pdo->prepare($sql);

        if(is_int(key($bindings)))
            return $this->bindQuestionPlaceholderParams($sql, $bindings);
        else
            return $this->bindNamedPlaceholderParams($sql, $bindings);
    }

    /**
     * Binds parameters where the questions marks are
     * @param  string       $sql        MySQL string
     * @param  array        $bindings   Array of needed parameters
     * @return PDOStatement             returns a PDOStatement that has binded parameters
     */
    protected function bindQuestionPlaceholderParams($sql, $bindings){


        $statement = $this->pdo->prepare($this->insertQuestionPlaceholderBindings($sql, $bindings));

        $index = 1;
        foreach ($bindings as $key => $value) {
            if(is_array($value)){
                foreach ($value as $k => $val) {
                    $this->bindParam($statement, $index++, $val);
                }
            }else{
                $this->bindParam($statement, $index++, $value);
            }
        }
        return $statement;
    }

    /**
     * Bings a singel param to the statment
     * @param  PDOStatement     &$statement     PDOStatement that is a reference
     * @param  mixed            $key            key to find the parameter
     * @param  mixed            $value          Parameter value
     */
    protected function bindParam(&$statement, $key, $value){
        $statement->bindParam($key, $value, \PDO::PARAM_NULL|\PDO::PARAM_INT|\PDO::PARAM_STR, 255);
    }

    /**
     * Binds named placed params to the statment
     * @param  string       $sql        MySQL string
     * @param  array        $bindings   Array of needed parameters
     * @return PDOStatement             returns a PDOStatement that has binded parameters
     */
    protected function bindNamedPlaceholderParams($sql, $bindings){
        $this->paramIndex = 1;
        $sql = $this->insertNamedPlaceholderBindings($sql, $bindings);

        $this->paramIndex = 1;
        $statement = $this->pdo->prepare($sql);
        foreach ($bindings as $key => $value) {
            if(is_array($value)){
                foreach ($value as $k => $val) {
                    $this->bindParam($statement, ":param_".$this->paramIndex++, $val);
                }
            }else{
                if(substr($key, 0, 1) == ":")
                    $this->bindParam($statement, $key, $value);
                else
                    $this->bindParam($statement, ":".$key, $value);
            }
        }
        return $statement;
    }

    /**
     * If a parameter is a array then we replace the quertion placeholder with the number of the array size as question marks
     * @param  string   $sql        MySQL string
     * @param  array    $bindings   Array of needed parameters
     * @return string               New MySQL string that has the right number of question marks
     */
    protected function insertQuestionPlaceholderBindings($sql, $bindings)
    {
        $finalSql = ''; // Stitched together. Question mark replaced with quoted values from $bindings.
        $offset = 0;

        foreach ($bindings as $value) {
            $newOffset = strpos($sql, '?', $offset); // Next '?'

            if ($newOffset === false)
                throw new \RuntimeException('Not enough placeholders!');

            $finalSql .= substr($sql, $offset, $newOffset-$offset);

            $finalSql .= $this->quoteArray($value);

            $offset = $newOffset+1; // Skip question mark placeholder.
        }

        // Last part (after the last placeholder).
        $finalSql .= substr($sql, $offset);

        return $finalSql;
    }

    /**
     * Replaces array with question marks
     * @param  mixed    $arr    Array of bindings
     * @return string           Returns a new MySQL string that has the correct format to add bindings
     */
    protected function quoteArray($arr)
    {
        // If not an array, just quote it.
        if (! is_array($arr))
            return '?';

        // Else, quote elements inside the array (first-level only),
        // and wrap quoted elements with parenthesis.

        $numElems = count($arr);
        if ($numElems === 0)
            return '()';

        $quoted = '(';
        for ($i = 0; $i < $numElems-1; $i++)
            $quoted .= '?,';

        $quoted .= '?)';
        return $quoted;
    }

    /**
     * Replaces array with the right named params
     * @param  mixed    $arr    Array of bindings
     * @param  mixed    $bind   Bind element for name
     * @return string           Returns a new MySQL string that has the correct format to add bindings
     */
    protected function quoteNamedArray($arr, $bind){
        // If not an array, just quote it.
        if (! is_array($arr)){
            if(substr($bind, 0, 1) == ":")
                return $bind;
            return ":".$bind;
        }

        // Else, quote elements inside the array (first-level only),
        // and wrap quoted elements with parenthesis.

        $numElems = count($arr);
        if ($numElems === 0)
            return '()';

        $quoted = '(';
        for ($i = 0; $i < $numElems-1; $i++)
            $quoted .= ':param_'.$this->paramIndex++.',';

        $quoted .= ':param_'.$this->paramIndex++.')';
        return $quoted;
    }

    /**
     * Creates a new MySQL string that replaces named params with the right names
     * @param  string   $sql        MySQL string
     * @param  array    $bindings   Array of needed parameters
     * @return string               Returns a new MySQL string that has the correct format to add bindings
     */
    protected function insertNamedPlaceholderBindings($sql, $bindings)
    {
        return preg_replace_callback(
            '/:\w+/',
            function($matches) use (&$bindings) {
                $bind = $matches[0];
                if (! array_key_exists($bind, $bindings)) // Support both [':id' => 42]  and  ['id' => 42]
                    $bind = substr($bind, 1);

                if (! array_key_exists($bind, $bindings))
                    throw new \RuntimeException('Placeholder ' . $bind . ' not found!');

                $value = $bindings[$bind];
                $replacement = $this->quoteNamedArray($value, $bind);

                unset($bindings[$bind]); // Imitate PDO behaviour - named placeholder cannot be repeated.
                return $replacement;
            },
            $sql
        );
    }
}
