<?php

namespace App\Services;

use App\Repositories\QuestionRepository;
use App\Repositories\ProjectRepository;
use App\Repositories\StructureTableRepository;

use App\Entities\Answer;
use App\Entities\Question;
use App\Entities\QuestionType;
use App\Entities\LanguageTextType;
use App\Services\StructureTableService;
use App\Services\AnswersConvertService;
use App\Exceptions\RuntimeException;

class QuestionService
{
    public function __construct(
      QuestionRepository    $questionRepository,
      ProjectRepository     $projectRepository,
      WarpitConfigService   $warpitConfigService,
      StructureTableService $structureTableService,
      AnswersConvertService $answersConvertService
      ) {
        $this->questionRepository    = $questionRepository;
        $this->projectRepository     = $projectRepository;
        $this->warpitConfigService   = $warpitConfigService;
        $this->structureTableService = $structureTableService;
        $this->answersConvertService = $answersConvertService;
    }

    public function create($questionName, $type, $projectName, $languageId)
    {
        if ($this->questionExists($questionName, $projectName, $languageId)) {
            throw new RuntimeException("Question name already exists!", 400);
        }

        $treeOrder    = $this->questionRepository->getMaxOrderForTreeTable($projectName)    + 1;
        $projectOrder = $this->questionRepository->getMaxOrderForProjectTable($projectName) + 1;

        $questionId = $this
                        ->questionRepository
                        ->insertTreeTable($questionName, $projectName, $treeOrder);

        $question            = $this->getDefaultValues($type);
        $question['id']      = $questionId;
        $question['q_pos']   = $projectOrder;
        $question['q_name']  = $questionName;
        $question['tip_sql'] = $type;

        $this
          ->questionRepository
          ->insertProjectTable($question, $projectName);

        $insertedQuestion = $this->getIfExists($questionName, $projectName, $languageId);

        return $insertedQuestion;
    }

    public function move($questionName, $projectName, $newPosition, $subFolder = 0)
    {
    }


    public function addQuestionText($questionName, $projectName, $text, $languageId)
    {
        $question    = $this->getIfExists($questionName, $projectName, $languageId);

        $oldText = $this->questionRepository->getFromLanguageTable($question->id, $projectName, LanguageTextType::QUESTION_TEXT, $languageId);

        if ($oldText === null) {
            return $this->questionRepository->insertIntoLanguageTable($question->id, $projectName, LanguageTextType::QUESTION_TEXT, $text, $languageId);
        }

        return $this->questionRepository->updateLanguageTable($question->id, $projectName, LanguageTextType::QUESTION_TEXT, $text, $languageId);
    }


    public function getPropertyValue($questionName, $projectName, $property, $languageId)
    {
        $question = $this->getIfExists($questionName, $projectName, $languageId);

        if (!property_exists($question, $property) || $question === null) {
            throw new RuntimeException("Property [$property] doesn't exists!", 400);
        }

        return $question->$property;
    }

    public function getQuestionText($questionName, $projectName, $languageId)
    {
        return $this->getPropertyValue($questionName, $projectName, 'questionText', $languageId);
    }

    public function delete($questionName, $projectName, $languageId)
    {
        $question    = $this->getIfExists($questionName, $projectName, $languageId);

        $questionId  = $question->id;

        $this->questionRepository->deleteFromProjectTable($questionName, $projectName);
        $this->questionRepository->deleteFromTreeTable($questionName, $projectName);
        $this->questionRepository->emptyLanguageTable($questionId, $projectName);
        //TODO delete from structure if exists, delete quotas, etc
    }

    public function questionExists($questionName, $projectName, $languageId)
    {
        return $this->getIfExists($questionName, $projectName, $languageId) !== null;
    }

    public function get($questionName, $projectName, $languageId)
    {
        $questionsData = $this->questionRepository->get($questionName, $projectName, $languageId);

        //TODO move this to the seperate transformer
        if (count($questionsData) <= 0) {
            return null;
        }

        if (count($questionsData) == 1) {
            return Question::createQuestion($questionsData->first());
        }

        $questions = [];
        foreach ($questionsData as $question) {
            $questions[] = Question::createQuestion($question);
        }

        return $questions;
    }

    public function getIfExists($questionName, $projectName, $languageId)
    {
        $question = $this->get($questionName, $projectName, $languageId);

        if($question === null){
          throw new RuntimeException("Question doesnt exists", 400);
        }

        return $question;
    }

    protected function getDefaultValues($type)
    {
        $valuesArray[QuestionType::SINGLE_RESPONSE] = [
                                                        'variableTypeBase' => 4,
                                                        'les_sql'          => 1,
                                                        'open_sql'         => 1
                                                      ];
        $valuesArray[QuestionType::MULTI_RESPONSE] =  [
                                                          'variableTypeBase' => 4,
                                                          'les_sql'          => 4,
                                                          'open_sql'         => 1
                                                      ];

        if (!key_exists($type, $valuesArray)) {
            return [];
        }

        return $valuesArray[$type];
    }
}
