<?php

namespace App\Repositories;

use App\Database\Schema\ColumnsBlueprint;
use Carbon\Carbon;

class StructureTableRepository extends BaseRepository
{
    public function create($name, $columns)
    {
        $this
            ->webcatiBaseConnection
            ->getSchemaBuilder()
            ->create($name, function ($table) use ($columns) {
                foreach ($columns->getColumns() as $column) {
                    $attributes = $column->getAttributes();
                    $name = array_shift($attributes);
                    $type = array_shift($attributes);

                    $table->addColumn($name, $type, $attributes);
                }

                $table->engine    = 'MyISAM';
                $table->charset   = 'utf8';
                $table->collation = 'utf8_unicode_ci';
            });
    }

    public function drop($name)
    {
        $this->webcatiBaseConnection->getSchemaBuilder()->dropIfExists($name);
    }

    public function backup($name)
    {
        $newName = $name . '_C' . Carbon::now()->format('YmdHis');
        $this->webcatiBaseConnection->getSchemaBuilder()->rename($name, $newName);
    }

    public function getBaseColumns()
    {
        $fields = new ColumnsBlueprint;
        $fields->increments('id');
        $fields->integer('id_client')->nullable();
        $fields->tinyInteger('flag_cli')->nullable();
        $fields->smallInteger('user_id')->nullable();
        $fields->tinyInteger('user_type')->nullable();
        $fields->integer('id_loginSID')->nullable();
        $fields->tinyInteger('id_device')->nullable();
        $fields->tinyInteger('id_language')->unsigned()->nullable();
        $fields->tinyInteger('smpl_var')->nullable();
        $fields->tinyInteger('sp_adult')->nullable();
        $fields->tinyInteger('sp_man')->nullable();
        $fields->char('id_site', 250)->nullable();
        $fields->integer('status')->nullable();
        $fields->integer('status_rec')->nullable();
        $fields->integer('flag_bad')->default(99);
        $fields->integer('t_status')->unsigned()->nullable();
        $fields->integer('t_start')->unsigned()->nullable();
        $fields->integer('t_stop')->unsigned()->nullable();
        $fields->dateTime('d_status')->nullable();
        $fields->dateTime('d_start')->nullable();
        $fields->dateTime('d_stop')->nullable();
        $fields->integer('t_pause')->default(0);
        $fields->integer('t_diffs')->default(0);
        $fields->integer('QT_user_id')->nullable();

        $fields->index(['user_id', 'user_type']);
        $fields->index(['status', 'status_rec']);
        $fields->index('id_device');
        $fields->index('id_language');
        $fields->index('flag_bad');
        $fields->index('id_site');
        $fields->index('id_loginsSID');
        $fields->index('sp_man');

        return $fields;
    }

    public function addColumns($name, array $columnsList)
    {
        $columns = $this->buildFields($columnsList);

        $this
          ->webcatiBaseConnection
          ->getSchemaBuilder()
          ->table($name . '_structure', function ($table) use ($columns) {
              foreach ($columns->getColumns() as $column) {
                  $attributes = $column->getAttributes();
                  $name = array_shift($attributes);
                  $type = array_shift($attributes);

                  $table->addColumn($name, $type, $attributes);
              }
          });
    }

    public function dropColumns($name, array $columnsList)
    {
      $this
        ->webcatiBaseConnection
        ->getSchemaBuilder()
        ->table($name . '_structure', function ($table) use ($columnsList) {
            foreach ($columnsList as $column) {
                $table->dropColumn($column->name);
            }
        });
    }

    public function getColumns($name)
    {
        return $this
            ->webcatiBaseConnection
            ->getSchemaBuilder()
            ->getColumnListing($name);
    }

    public function quotaColumns()
    {
    }

    public function getEngineLogColumns()
    {
        $fields = new ColumnsBlueprint;
        $fields->increments('id');
        $fields->integer('id_structure');
        $fields->smallInteger('id_question');
        $fields->integer('id_user');
        $fields->tinyInteger('user_type');
        $fields->smallInteger('id_language')->nullable();
        $fields->string('q_name', 25);
        $fields->text('currentRotationString');
        $fields->text('questionnairePath');
        $fields->integer('engineTime');
        $fields->integer('questStart')->nullable();
        $fields->integer('questEnd')->nullable();
        $fields->tinyInteger('arriveStatus')->nullable();
        $fields->text('eventLogger')->nullable();
        $fields->index('id_structure');

        return $fields;
    }

    private function buildFields(array $columnList)
    {
        $fields = new ColumnsBlueprint;

        foreach ($columnList as $column) {
            switch ($column->dataType) {
                case 0: $fields->string($column->name, 255)->nullable();break;
                case 1: $fields->string($column->name, 150)->nullable();break;
                case 2: $fields->string($column->name, 50)->nullable();break;
                case 3: $fields->string($column->name, 50)->nullable();break;
                case 4: $fields->integer($column->name)->nullable();break;
                case 5: $fields->tinyInteger($column->name)->nullable();break;
                case 6: $fields->text($column->name)->nullable();break;
                case 7: $fields->boolean($column->name)->nullable();break;
                case 8: $fields->date($column->name)->nullable(); break;
                case 9: $fields->dateTime($column->name)->nullable(); break;
                case 10: $fields->double($column->name)->nullable(); break;
            }
        }

        return $fields;
    }

}
