<?php

namespace App\Repositories;

use App\Entities\LanguageTextType;

class QuestionRepository extends BaseRepository
{
    public function get($questionName, $projectName, $languageId)
    {
        return $this
                ->webcatiConnection
                ->table($projectName . ' as a')
                ->leftJoin($projectName . '_language as b', function ($join) use ($languageId) {
                    $join
                      ->on('a.id', '=', 'b.id_question')
                      ->where('b.id_text_type', '=', LanguageTextType::ANSWER_STRING)
                      ->where('b.id_language', '=', $languageId);
                })
                ->leftJoin($projectName . '_language as c', function ($join) use ($languageId) {
                    $join
                      ->on('a.id', '=', 'c.id_question')
                      ->where('c.id_text_type', '=', LanguageTextType::QUESTION_TEXT)
                      ->where('c.id_language', '=', $languageId);
                })
                ->where('q_name', $questionName)
                ->select('a.*', 'b.lngText as answersString', 'c.lngText as questionText')
                ->get();
    }

    public function getMaxOrderForTreeTable($projectName)
    {
        return $this
                ->webcatiConnection
                ->table($projectName . '_tree')
                ->select('orderBy')
                ->orderByDesc('orderBy')
                ->first()
                ->orderBy;
    }

    public function getMaxOrderForProjectTable($projectName)
    {
        return $this
              ->webcatiConnection
              ->table($projectName)
              ->select('q_pos')
              ->orderByDesc('q_pos')
              ->first()
              ->q_pos;
    }

    public function insertTreeTable($questionName, $projectName, $orderBy, $parent = 0)
    {
        $questionId = $this->webcatiConnection
                          ->table($projectName . '_tree')
                          ->insertGetId([
                            'id_sup'  => $parent,
                            'orderBy' => $orderBy,
                            'idLab'   => $questionName,
                            'is_leaf' => true,
                          ]);

        $this
            ->webcatiConnection
            ->table($projectName . '_tree')
            ->where('id', $questionId)
            ->update(['id_question' => $questionId]);

        return $questionId;
    }

    public function insertProjectTable($question, $projectName)
    {
        return $this->webcatiConnection
                    ->table($projectName)
                    ->insertGetId($question);
    }

    public function deleteFromTreeTable($questionName, $projectName)
    {
        $this
          ->webcatiConnection
          ->table($projectName)
          ->where('q_name', $questionName)
          ->delete();
    }

    public function deleteFromProjectTable($questionName, $projectName)
    {
        $this
        ->webcatiConnection
        ->table($projectName . '_tree')
        ->where('idLab', $questionName)
        ->delete();
    }

    public function emptyLanguageTable($questionId, $projectName)
    {
        $this
          ->webcatiConnection
          ->table($projectName . '_language')
          ->where('id_question', $questionId)
          ->delete();
    }

    public function insertIntoLanguageTable($questionId, $projectName, $textType, $label, $languageId)
    {
        $this->webcatiConnection->table($projectName . '_language')->insert([
          'id_language'  => $languageId,
          'id_question'  => $questionId,
          'id_text_type' => $textType,
          'lngText'      => $label,
        ]);
    }

    public function getFromLanguageTable($questionId, $projectName, $textType, $languageId)
    {
        $data =  $this->webcatiConnection
                ->table($projectName . '_language')
                ->where('id_question', $questionId)
                ->where('id_language', $languageId)
                ->where('id_text_type', $textType)
                ->select('lngText')
                ->first();

        if ($data===null) {
            return null;
        }


        return $data->lngText;
    }

    public function updateLanguageTable($questionId, $projectName, $textType, $label, $languageId)
    {
      $this
          ->webcatiConnection
          ->table($projectName . '_language')
          ->where('id_question',  $questionId)
          ->where('id_language',  $languageId)
          ->where('id_text_type', $textType)
          ->update(['lngText' => $label]);
    }

    public function deleteFromLanguageTable($questionId, $projectName, $textType, $languageId)
    {
      $this
          ->webcatiConnection
          ->table($projectName . '_language')
          ->where('id_question',  $questionId)
          ->where('id_language',  $languageId)
          ->where('id_text_type', $textType)
          ->delete();
    }
}
